const { DataTypes } = require('sequelize');
const { sequelize } = require('../config/database');

const Contract = sequelize.define('Contract', {
  id: {
    type: DataTypes.INTEGER,
    primaryKey: true,
    autoIncrement: true
  },
  contractNumber: {
    type: DataTypes.STRING(50),
    allowNull: false,
    unique: true
  },
  contractDate: {
    type: DataTypes.DATEONLY,
    allowNull: false,
    validate: {
      isDate: true
    }
  },
  secondParty: {
    type: DataTypes.STRING(200),
    allowNull: false,
    validate: {
      len: [2, 200],
      notEmpty: true
    }
  },
  location: {
    type: DataTypes.TEXT,
    allowNull: false,
    validate: {
      notEmpty: true
    }
  },
  ownerName: {
    type: DataTypes.STRING(100),
    allowNull: false,
    validate: {
      len: [2, 100],
      notEmpty: true
    }
  },
  ownerNationalID: {
    type: DataTypes.STRING(20),
    allowNull: false,
    validate: {
      len: [10, 20],
      isNumeric: true
    }
  },
  ownerPhoneNumber: {
    type: DataTypes.STRING(20),
    allowNull: false,
    validate: {
      len: [10, 20],
      isNumeric: true
    }
  },
  camerasType: {
    type: DataTypes.STRING(100),
    allowNull: false,
    validate: {
      notEmpty: true
    }
  },
  camerasResolution: {
    type: DataTypes.STRING(50),
    allowNull: false,
    validate: {
      notEmpty: true
    }
  },
  innerCamerasNumber: {
    type: DataTypes.INTEGER,
    allowNull: false,
    defaultValue: 0,
    validate: {
      min: 0
    }
  },
  outerCamerasNumber: {
    type: DataTypes.INTEGER,
    allowNull: false,
    defaultValue: 0,
    validate: {
      min: 0
    }
  },
  storageCapacity: {
    type: DataTypes.TEXT,
    allowNull: false,
    validate: {
      notEmpty: true
    }
  },
  startsFrom: {
    type: DataTypes.DATEONLY,
    allowNull: false,
    validate: {
      isDate: true
    }
  },
  endsAt: {
    type: DataTypes.DATEONLY,
    allowNull: false,
    validate: {
      isDate: true,
      isAfterStartDate(value) {
        if (value <= this.startsFrom) {
          throw new Error('تاريخ انتهاء العقد يجب أن يكون بعد تاريخ البداية');
        }
      }
    }
  },
  attachments: {
    type: DataTypes.JSON,
    allowNull: true,
    defaultValue: []
  },
  notes: {
    type: DataTypes.TEXT,
    allowNull: true
  },
  status: {
    type: DataTypes.ENUM('active', 'expired', 'cancelled'),
    defaultValue: 'active'
  },
  createdBy: {
    type: DataTypes.INTEGER,
    allowNull: false,
    references: {
      model: 'users',
      key: 'id'
    }
  },
  updatedBy: {
    type: DataTypes.INTEGER,
    allowNull: true,
    references: {
      model: 'users',
      key: 'id'
    }
  }
}, {
  tableName: 'contracts',
  timestamps: true,
  indexes: [
    {
      fields: ['contractNumber']
    },
    {
      fields: ['contractDate']
    },
    {
      fields: ['secondParty']
    },
    {
      fields: ['status']
    },
    {
      fields: ['startsFrom', 'endsAt']
    }
  ]
});

// Virtual field for total cameras
Contract.prototype.getTotalCameras = function() {
  return this.innerCamerasNumber + this.outerCamerasNumber;
};

// Virtual field for contract duration in days
Contract.prototype.getContractDuration = function() {
  const start = new Date(this.startsFrom);
  const end = new Date(this.endsAt);
  const diffTime = Math.abs(end - start);
  return Math.ceil(diffTime / (1000 * 60 * 60 * 24));
};

module.exports = Contract;
