const express = require('express');
const { body, validationResult } = require('express-validator');
const { Settings } = require('../models');
const { authenticateToken, requireAdmin } = require('../middleware/auth');
const { sequelize } = require('../config/database');
const path = require('path');
const fs = require('fs').promises;
const { spawn } = require('child_process');
const cron = require('node-cron');
const moment = require('moment');
const mysqldump = require('mysqldump');
const mysql = require('mysql2/promise');

const router = express.Router();

// 1. SECURE DATABASE BACKUP - Using Node.js mysqldump package
async function createDatabaseBackup() {
  try {
    const backupDir = path.join(__dirname, '../backups');
    await fs.mkdir(backupDir, { recursive: true });

    const timestamp = moment().format('YYYY-MM-DD_HH-mm-ss');
    const backupFileName = `backup_${timestamp}.sql`;
    const backupPath = path.join(backupDir, backupFileName);

    // Database connection configuration
    const connectionConfig = {
      host: process.env.DB_HOST || 'localhost',
      port: parseInt(process.env.DB_PORT) || 3306,
      user: process.env.DB_USER || 'root',
      password: process.env.DB_PASSWORD || '',
      database: process.env.DB_NAME || 'contracts_maintenance'
    };

    // Dump configuration
    const dumpConfig = {
      connection: connectionConfig,
      dumpToFile: backupPath,
      compressFile: false,
      dump: {
        schema: {
          format: true,
          autoIncrement: true,
          engine: true,
          table: {
            ifNotExist: true,
            dropIfExist: true,
            charset: true
          },
          view: {
            createOrReplace: true
          }
        },
        data: {
          format: true,
          verbose: false,
          lockTables: false,
          includeViewData: false
        },
        trigger: {
          delimiter: ';;'
        }
      }
    };

    // Create the backup
    const result = await mysqldump(dumpConfig);
    
    // Verify backup file was created and has content
    const stats = await fs.stat(backupPath);
    if (stats.size < 1000) { // Less than 1KB is suspicious
      await fs.unlink(backupPath);
      throw new Error('Backup file is too small - likely failed');
    }

    return {
      fileName: backupFileName,
      filePath: backupPath,
      size: stats.size,
      createdAt: new Date()
    };

  } catch (error) {
    throw new Error(`Backup creation failed: ${error.message}`);
  }
}

// 2. SECURE FILE VALIDATION with additional checks
function validateAndSanitizeFileName(fileName) {
  // Remove any path components
  const baseName = path.basename(fileName);
  
  // Strict validation
  if (!baseName.match(/^backup_\d{4}-\d{2}-\d{2}_\d{2}-\d{2}-\d{2}\.sql$/)) {
    throw new Error('Invalid backup filename format');
  }
  
  return baseName;
}

// Helper function to restore database from SQL file
async function restoreDatabaseFromFile(filePath) {
  try {
    // Read the SQL file
    const sqlContent = await fs.readFile(filePath, 'utf8');
    
    // Split SQL content into individual statements
    const statements = sqlContent
      .split(';')
      .map(stmt => stmt.trim())
      .filter(stmt => stmt.length > 0 && !stmt.startsWith('--') && !stmt.startsWith('/*'));

    // Database connection configuration
    const connectionConfig = {
      host: process.env.DB_HOST || 'localhost',
      port: parseInt(process.env.DB_PORT) || 3306,
      user: process.env.DB_USER || 'root',
      password: process.env.DB_PASSWORD || '',
      database: process.env.DB_NAME || 'contracts_maintenance',
      multipleStatements: true
    };

    // Create connection
    const connection = await mysql.createConnection(connectionConfig);

    try {
      // Execute each statement
      for (const statement of statements) {
        if (statement.trim()) {
          await connection.execute(statement);
        }
      }
      
      await connection.end();
      return { success: true };
      
    } catch (error) {
      await connection.end();
      throw error;
    }

  } catch (error) {
    throw new Error(`Database restore failed: ${error.message}`);
  }
}

// Helper function to clean old backups
async function cleanOldBackups() {
  try {
    const backupDir = path.join(__dirname, '../backups');
    const retentionDays = await Settings.getValue('backup_retention_days', 30);
    const cutoffDate = moment().subtract(retentionDays, 'days').toDate();

    const files = await fs.readdir(backupDir);
    
    for (const file of files) {
      if (file.startsWith('backup_') && file.endsWith('.sql')) {
        const filePath = path.join(backupDir, file);
        const stats = await fs.stat(filePath);
        
        if (stats.mtime < cutoffDate) {
          await fs.unlink(filePath);
          console.log(`Deleted old backup: ${file}`);
        }
      }
    }
  } catch (error) {
    console.error('Error cleaning old backups:', error);
  }
}

// 4. SAFE SCHEDULER INITIALIZATION
let backupScheduler = null;

async function initializeBackupScheduler() {
  try {
    // Wait for database to be ready
    await sequelize.authenticate();
    
    if (backupScheduler) {
      backupScheduler.stop();
      backupScheduler = null;
    }

    const backupEnabled = await Settings.getValue('backup_enabled', true);
    const backupSchedule = await Settings.getValue('backup_schedule', '0 2 * * *');

    if (backupEnabled && cron.validate(backupSchedule)) {
      backupScheduler = cron.schedule(backupSchedule, async () => {
        try {
          console.log('Starting scheduled backup...');
          await createDatabaseBackup();
          await cleanOldBackups();
          console.log('Scheduled backup completed successfully');
        } catch (error) {
          console.error('Scheduled backup failed:', error);
          // TODO: Send alert to admin
        }
      }, {
        scheduled: true,
        timezone: 'Asia/Riyadh'
      });

      console.log(`Backup scheduler initialized: ${backupSchedule}` );
    }
  } catch (error) {
    console.error('Error initializing backup scheduler:', error);
    // Don't throw - allow app to start even if scheduler fails
  }
}

// Create manual backup
router.post('/create', [
  authenticateToken,
  requireAdmin
], async (req, res) => {
  try {
    const backup = await createDatabaseBackup();
    
    res.json({
      success: true,
      message: 'تم إنشاء النسخة الاحتياطية بنجاح',
      data: backup
    });
  } catch (error) {
    console.error('Create backup error:', error);
    res.status(500).json({
      success: false,
      message: 'خطأ في إنشاء النسخة الاحتياطية',
      error: error.message
    });
  }
});

// List all backups
router.get('/list', [
  authenticateToken,
  requireAdmin
], async (req, res) => {
  try {
    const backupDir = path.join(__dirname, '../backups');
    
    try {
      await fs.access(backupDir);
    } catch {
      await fs.mkdir(backupDir, { recursive: true });
    }

    const files = await fs.readdir(backupDir);
    const backups = [];

    for (const file of files) {
      if (file.startsWith('backup_') && file.endsWith('.sql')) {
        const filePath = path.join(backupDir, file);
        const stats = await fs.stat(filePath);
        
        backups.push({
          fileName: file,
          size: stats.size,
          createdAt: stats.mtime,
          sizeFormatted: formatFileSize(stats.size)
        });
      }
    }

    // Sort by creation date (newest first)
    backups.sort((a, b) => new Date(b.createdAt) - new Date(a.createdAt));

    res.json({
      success: true,
      data: backups
    });
  } catch (error) {
    console.error('List backups error:', error);
    res.status(500).json({
      success: false,
      message: 'خطأ في جلب قائمة النسخ الاحتياطية'
    });
  }
});

// Download backup
router.get('/download/:fileName', [
  authenticateToken,
  requireAdmin
], async (req, res) => {
  try {
    // Secure filename validation
    const fileName = validateAndSanitizeFileName(req.params.fileName);
    if (!fileName) {
      return res.status(400).json({
        success: false,
        message: 'اسم الملف غير صحيح'
      });
    }

    const filePath = path.join(__dirname, '../backups', fileName);
    
    try {
      await fs.access(filePath);
    } catch {
      return res.status(404).json({
        success: false,
        message: 'الملف غير موجود'
      });
    }

    res.setHeader('Content-Type', 'application/octet-stream');
    res.setHeader('Content-Disposition', `attachment; filename="${fileName}"`);
    
    const fileStream = require('fs').createReadStream(filePath);
    fileStream.pipe(res);
  } catch (error) {
    console.error('Download backup error:', error);
    res.status(500).json({
      success: false,
      message: 'خطأ في تحميل النسخة الاحتياطية'
    });
  }
});

// Delete backup
router.delete('/:fileName', [
  authenticateToken,
  requireAdmin
], async (req, res) => {
  try {
    // Secure filename validation
    const fileName = validateAndSanitizeFileName(req.params.fileName);
    if (!fileName) {
      return res.status(400).json({
        success: false,
        message: 'اسم الملف غير صحيح'
      });
    }

    const filePath = path.join(__dirname, '../backups', fileName);
    
    try {
      await fs.unlink(filePath);
      res.json({
        success: true,
        message: 'تم حذف النسخة الاحتياطية بنجاح'
      });
    } catch (error) {
      if (error.code === 'ENOENT') {
        return res.status(404).json({
          success: false,
          message: 'الملف غير موجود'
        });
      }
      throw error;
    }
  } catch (error) {
    console.error('Delete backup error:', error);
    res.status(500).json({
      success: false,
      message: 'خطأ في حذف النسخة الاحتياطية'
    });
  }
});

// 3. FIXED RESTORE ENDPOINT - Proper async handling
router.post('/restore/:fileName', [
  authenticateToken,
  requireAdmin,
  body('confirmRestore').equals('true').withMessage('يجب تأكيد الاستعادة')
], async (req, res) => {
  try {
    const errors = validationResult(req);
    if (!errors.isEmpty()) {
      return res.status(400).json({
        success: false,
        message: 'بيانات غير صحيحة',
        errors: errors.array()
      });
    }

    // Secure filename validation
    const fileName = validateAndSanitizeFileName(req.params.fileName);
    const filePath = path.join(__dirname, '../backups', fileName);
    
    try {
      await fs.access(filePath);
    } catch {
      return res.status(404).json({
        success: false,
        message: 'الملف غير موجود'
      });
    }

    const dbConfig = {
      host: process.env.DB_HOST,
      port: process.env.DB_PORT,
      database: process.env.DB_NAME,
      username: process.env.DB_USER
    };

    // Perform restore using Node.js
    const restoreResult = await restoreDatabaseFromFile(filePath);

    // Response sent AFTER promise resolves
    res.json({
      success: true,
      message: 'تم استعادة النسخة الاحتياطية بنجاح'
    });

  } catch (error) {
    console.error('Restore backup error:', error);
    
    if (!res.headersSent) {
      res.status(500).json({
        success: false,
        message: 'خطأ في استعادة النسخة الاحتياطية',
        error: process.env.NODE_ENV === 'development' ? error.message : undefined
      });
    }
  }
});

// Update backup settings
router.put('/settings', [
  authenticateToken,
  requireAdmin,
  body('backupEnabled').optional().isBoolean(),
  body('backupSchedule').optional().custom((value) => {
    if (!cron.validate(value)) {
      throw new Error('جدولة النسخ الاحتياطي غير صحيحة');
    }
    return true;
  }),
  body('retentionDays').optional().isInt({ min: 1, max: 365 })
], async (req, res) => {
  try {
    const errors = validationResult(req);
    if (!errors.isEmpty()) {
      return res.status(400).json({
        success: false,
        message: 'بيانات غير صحيحة',
        errors: errors.array()
      });
    }

    const { backupEnabled, backupSchedule, retentionDays } = req.body;

    if (backupEnabled !== undefined) {
      await Settings.setValue('backup_enabled', backupEnabled, 'boolean', req.user.id);
    }

    if (backupSchedule) {
      await Settings.setValue('backup_schedule', backupSchedule, 'string', req.user.id);
    }

    if (retentionDays) {
      await Settings.setValue('backup_retention_days', retentionDays, 'number', req.user.id);
    }

    // Reinitialize scheduler with new settings
    await initializeBackupScheduler();

    res.json({
      success: true,
      message: 'تم تحديث إعدادات النسخ الاحتياطي بنجاح'
    });
  } catch (error) {
    console.error('Update backup settings error:', error);
    res.status(500).json({
      success: false,
      message: 'خطأ في تحديث إعدادات النسخ الاحتياطي'
    });
  }
});

// Get backup settings
router.get('/settings', [
  authenticateToken,
  requireAdmin
], async (req, res) => {
  try {
    const backupEnabled = await Settings.getValue('backup_enabled', true);
    const backupSchedule = await Settings.getValue('backup_schedule', '0 2 * * *');
    const retentionDays = await Settings.getValue('backup_retention_days', 30);

    res.json({
      success: true,
      data: {
        backupEnabled,
        backupSchedule,
        retentionDays,
        isSchedulerRunning: backupScheduler ? backupScheduler.running : false
      }
    });
  } catch (error) {
    console.error('Get backup settings error:', error);
    res.status(500).json({
      success: false,
      message: 'خطأ في جلب إعدادات النسخ الاحتياطي'
    });
  }
});

// Restore from latest backup
router.post('/restore-latest', [
  authenticateToken,
  requireAdmin,
  body('confirmRestore').equals('true').withMessage('يجب تأكيد الاستعادة')
], async (req, res) => {
  try {
    const errors = validationResult(req);
    if (!errors.isEmpty()) {
      return res.status(400).json({
        success: false,
        message: 'بيانات غير صحيحة',
        errors: errors.array()
      });
    }

    const backupDir = path.join(__dirname, '../backups');
    
    try {
      await fs.access(backupDir);
    } catch {
      return res.status(404).json({
        success: false,
        message: 'لا توجد نسخ احتياطية متاحة'
      });
    }

    const files = await fs.readdir(backupDir);
    const backupFiles = files.filter(file => 
      file.startsWith('backup_') && file.endsWith('.sql')
    );

    if (backupFiles.length === 0) {
      return res.status(404).json({
        success: false,
        message: 'لا توجد نسخ احتياطية متاحة'
      });
    }

    // Sort by filename (which contains timestamp) to get the latest
    backupFiles.sort((a, b) => b.localeCompare(a));
    const latestBackup = backupFiles[0];
    const filePath = path.join(backupDir, latestBackup);

    // Perform restore using Node.js
    const restoreResult = await restoreDatabaseFromFile(filePath);

    res.json({
      success: true,
      message: `تم استعادة النسخة الاحتياطية الأحدث بنجاح (${latestBackup})`,
      restoredBackup: latestBackup
    });
  } catch (error) {
    console.error('Restore latest backup error:', error);
    res.status(500).json({
      success: false,
      message: 'خطأ في استعادة النسخة الاحتياطية'
    });
  }
});

// Create and download backup
router.post('/create-and-download', [
  authenticateToken,
  requireAdmin
], async (req, res) => {
  try {
    // First create a new backup
    const backup = await createDatabaseBackup();
    
    // Then send it as download
    const filePath = backup.filePath;
    
    res.setHeader('Content-Type', 'application/octet-stream');
    res.setHeader('Content-Disposition', `attachment; filename="${backup.fileName}"`);
    res.setHeader('Content-Length', backup.size);
    
    const fileStream = require('fs').createReadStream(filePath);
    fileStream.pipe(res);
    
    fileStream.on('error', (error) => {
      console.error('File stream error:', error);
      if (!res.headersSent) {
        res.status(500).json({
          success: false,
          message: 'خطأ في تحميل النسخة الاحتياطية'
        });
      }
    });
    
  } catch (error) {
    console.error('Create and download backup error:', error);
    if (!res.headersSent) {
      res.status(500).json({
        success: false,
        message: 'خطأ في إنشاء وتحميل النسخة الاحتياطية',
        error: error.message
      });
    }
  }
});

// Clean old backups manually
router.post('/clean', [
  authenticateToken,
  requireAdmin
], async (req, res) => {
  try {
    await cleanOldBackups();
    
    res.json({
      success: true,
      message: 'تم تنظيف النسخ الاحتياطية القديمة بنجاح'
    });
  } catch (error) {
    console.error('Clean backups error:', error);
    res.status(500).json({
      success: false,
      message: 'خطأ في تنظيف النسخ الاحتياطية'
    });
  }
});

// Helper function to format file size
function formatFileSize(bytes) {
  if (bytes === 0) return '0 Bytes';
  
  const k = 1024;
  const sizes = ['Bytes', 'KB', 'MB', 'GB'];
  const i = Math.floor(Math.log(bytes) / Math.log(k));
  
  return parseFloat((bytes / Math.pow(k, i)).toFixed(2)) + ' ' + sizes[i];
}

// 5. Initialize scheduler AFTER app is ready (in main app.js)
// Don't call immediately on module load
// Instead export the function and call it after database connection:
// await initializeBackupScheduler();

module.exports = { router, initializeBackupScheduler };
