const express = require('express');
const { body, validationResult, query } = require('express-validator');
const { Contract, User, Scene, Settings } = require('../models');
const { authenticateToken, checkPermission } = require('../middleware/auth');
const { Op } = require('sequelize');
const multer = require('multer');
const path = require('path');
const fs = require('fs').promises;

const router = express.Router();

// Configure multer for file uploads
const storage = multer.diskStorage({
  destination: async (req, file, cb) => {
    const uploadPath = path.join(__dirname, '../uploads/contracts');
    try {
      await fs.mkdir(uploadPath, { recursive: true });
      cb(null, uploadPath);
    } catch (error) {
      cb(error);
    }
  },
  filename: (req, file, cb) => {
    const uniqueSuffix = Date.now() + '-' + Math.round(Math.random() * 1E9);
    cb(null, `contract-${uniqueSuffix}${path.extname(file.originalname)}`);
  }
});

const upload = multer({
  storage,
  limits: {
    fileSize: parseInt(process.env.MAX_FILE_SIZE) || 10485760 // 10MB
  },
  fileFilter: (req, file, cb) => {
    const allowedTypes = /jpeg|jpg|png|gif|pdf|doc|docx|xls|xlsx/;
    const extname = allowedTypes.test(path.extname(file.originalname).toLowerCase());
    const mimetype = allowedTypes.test(file.mimetype);

    if (mimetype && extname) {
      return cb(null, true);
    } else {
      cb(new Error('نوع الملف غير مدعوم'));
    }
  }
});

// Get next contract number
router.get('/next-number', [
  authenticateToken,
  checkPermission('contracts', 'add')
], async (req, res) => {
  try {
    const lastContract = await Contract.findOne({
      order: [['contractNumber', 'DESC']],
      attributes: ['contractNumber']
    });

    let nextNumber = 1;
    if (lastContract) {
      const lastNumber = parseInt(lastContract.contractNumber.replace(/\D/g, ''));
      nextNumber = lastNumber + 1;
    }

    // Check if there's a custom next number in settings
    const customNextNumber = await Settings.getValue('next_contract_number');
    if (customNextNumber && customNextNumber > nextNumber) {
      nextNumber = customNextNumber;
    }

    const formattedNumber = `C-${nextNumber.toString().padStart(6, '0')}`;

    res.json({
      success: true,
      data: { nextNumber: formattedNumber }
    });
  } catch (error) {
    console.error('Get next contract number error:', error);
    res.status(500).json({
      success: false,
      message: 'خطأ في جلب رقم العقد التالي'
    });
  }
});

// Get all contracts
router.get('/', [
  authenticateToken,
  checkPermission('contracts', 'view'),
  query('page').optional().isInt({ min: 1 }),
  query('limit').optional().isInt({ min: 1, max: 100 }),
  query('search').optional().isString(),
  query('status').optional().isIn(['active', 'expired', 'cancelled']),
  query('dateFrom').optional().isISO8601(),
  query('dateTo').optional().isISO8601()
], async (req, res) => {
  try {
    const errors = validationResult(req);
    if (!errors.isEmpty()) {
      return res.status(400).json({
        success: false,
        message: 'بيانات غير صحيحة',
        errors: errors.array()
      });
    }

    const page = parseInt(req.query.page) || 1;
    const limit = parseInt(req.query.limit) || 10;
    const search = req.query.search || '';
    const status = req.query.status;
    const dateFrom = req.query.dateFrom;
    const dateTo = req.query.dateTo;
    const offset = (page - 1) * limit;

    let whereClause = {};

    // Search filter
    if (search) {
      whereClause[Op.or] = [
        { contractNumber: { [Op.like]: `%${search}%` } },
        { secondParty: { [Op.like]: `%${search}%` } },
        { ownerName: { [Op.like]: `%${search}%` } },
        { location: { [Op.like]: `%${search}%` } }
      ];
    }

    // Status filter
    if (status) {
      whereClause.status = status;
    }

    // Date range filter
    if (dateFrom && dateTo) {
      whereClause.contractDate = {
        [Op.between]: [dateFrom, dateTo]
      };
    } else if (dateFrom) {
      whereClause.contractDate = {
        [Op.gte]: dateFrom
      };
    } else if (dateTo) {
      whereClause.contractDate = {
        [Op.lte]: dateTo
      };
    }

    const { count, rows } = await Contract.findAndCountAll({
      where: whereClause,
      include: [
        {
          model: User,
          as: 'creator',
          attributes: ['id', 'fullName']
        },
        {
          model: Scene,
          as: 'scenes',
          attributes: ['id', 'date', 'action', 'status']
        }
      ],
      limit,
      offset,
      order: [['createdAt', 'DESC']]
    });

    res.json({
      success: true,
      data: {
        contracts: rows,
        pagination: {
          currentPage: page,
          totalPages: Math.ceil(count / limit),
          totalItems: count,
          itemsPerPage: limit
        }
      }
    });
  } catch (error) {
    console.error('Get contracts error:', error);
    res.status(500).json({
      success: false,
      message: 'خطأ في جلب العقود'
    });
  }
});

// Get contract by ID
router.get('/:id', [
  authenticateToken,
  checkPermission('contracts', 'view')
], async (req, res) => {
  try {
    const contract = await Contract.findByPk(req.params.id, {
      include: [
        {
          model: User,
          as: 'creator',
          attributes: ['id', 'fullName']
        },
        {
          model: User,
          as: 'updater',
          attributes: ['id', 'fullName']
        },
        {
          model: Scene,
          as: 'scenes',
          include: [
            {
              model: User,
              as: 'creator',
              attributes: ['id', 'fullName']
            }
          ],
          order: [['date', 'DESC']]
        }
      ]
    });

    if (!contract) {
      return res.status(404).json({
        success: false,
        message: 'العقد غير موجود'
      });
    }

    res.json({
      success: true,
      data: contract
    });
  } catch (error) {
    console.error('Get contract error:', error);
    res.status(500).json({
      success: false,
      message: 'خطأ في جلب العقد'
    });
  }
});

// Create new contract
router.post('/', [
  authenticateToken,
  checkPermission('contracts', 'add'),
  upload.array('attachments', 10),
  body('contractNumber').notEmpty().withMessage('رقم العقد مطلوب'),
  body('contractDate').isISO8601().withMessage('تاريخ العقد غير صحيح'),
  body('secondParty').isLength({ min: 2, max: 200 }).withMessage('الطرف الثاني مطلوب'),
  body('location').notEmpty().withMessage('الموقع مطلوب'),
  body('ownerName').isLength({ min: 2, max: 100 }).withMessage('اسم المالك مطلوب'),
  body('ownerNationalID').isLength({ min: 10, max: 20 }).isNumeric().withMessage('رقم الهوية غير صحيح'),
  body('ownerPhoneNumber').isLength({ min: 10, max: 20 }).isNumeric().withMessage('رقم الهاتف غير صحيح'),
  body('camerasType').notEmpty().withMessage('نوع الكاميرات مطلوب'),
  body('camerasResolution').notEmpty().withMessage('دقة الكاميرات مطلوبة'),
  body('innerCamerasNumber').isInt({ min: 0 }).withMessage('عدد الكاميرات الداخلية غير صحيح'),
  body('outerCamerasNumber').isInt({ min: 0 }).withMessage('عدد الكاميرات الخارجية غير صحيح'),
  body('storageCapacity').notEmpty().withMessage('سعة التخزين مطلوبة'),
  body('startsFrom').isISO8601().withMessage('تاريخ البداية غير صحيح'),
  body('endsAt').isISO8601().withMessage('تاريخ النهاية غير صحيح')
], async (req, res) => {
  try {
    const errors = validationResult(req);
    if (!errors.isEmpty()) {
      return res.status(400).json({
        success: false,
        message: 'بيانات غير صحيحة',
        errors: errors.array()
      });
    }

    // Check if contract number already exists
    const existingContract = await Contract.findOne({
      where: { contractNumber: req.body.contractNumber }
    });

    if (existingContract) {
      return res.status(400).json({
        success: false,
        message: 'رقم العقد موجود بالفعل'
      });
    }

    // Validate dates
    if (new Date(req.body.endsAt) <= new Date(req.body.startsFrom)) {
      return res.status(400).json({
        success: false,
        message: 'تاريخ انتهاء العقد يجب أن يكون بعد تاريخ البداية'
      });
    }

    // Process attachments
    const attachments = [];
    if (req.files && req.files.length > 0) {
      for (const file of req.files) {
        attachments.push({
          name: file.originalname,
          attachment: file.filename,
          dateOfUpload: new Date(),
          uploadedBy: req.user.id
        });
      }
    }

    const contractData = {
      ...req.body,
      attachments,
      createdBy: req.user.id
    };

    const contract = await Contract.create(contractData);

    // Update next contract number setting if needed
    const currentNumber = parseInt(req.body.contractNumber.replace(/\D/g, ''));
    await Settings.setValue('next_contract_number', currentNumber + 1, 'number', req.user.id);

    res.status(201).json({
      success: true,
      message: 'تم إنشاء العقد بنجاح',
      data: contract
    });
  } catch (error) {
    console.error('Create contract error:', error);
    res.status(500).json({
      success: false,
      message: 'خطأ في إنشاء العقد'
    });
  }
});

// Update contract
router.put('/:id', [
  authenticateToken,
  checkPermission('contracts', 'edit'),
  upload.array('newAttachments', 10),
  body('contractDate').optional().isISO8601(),
  body('secondParty').optional().isLength({ min: 2, max: 200 }),
  body('location').optional().notEmpty(),
  body('ownerName').optional().isLength({ min: 2, max: 100 }),
  body('ownerNationalID').optional().isLength({ min: 10, max: 20 }).isNumeric(),
  body('ownerPhoneNumber').optional().isLength({ min: 10, max: 20 }).isNumeric(),
  body('camerasType').optional().notEmpty(),
  body('camerasResolution').optional().notEmpty(),
  body('innerCamerasNumber').optional().isInt({ min: 0 }),
  body('outerCamerasNumber').optional().isInt({ min: 0 }),
  body('storageCapacity').optional().notEmpty(),
  body('startsFrom').optional().isISO8601(),
  body('endsAt').optional().isISO8601(),
  body('status').optional().isIn(['active', 'expired', 'cancelled'])
], async (req, res) => {
  try {
    const errors = validationResult(req);
    if (!errors.isEmpty()) {
      return res.status(400).json({
        success: false,
        message: 'بيانات غير صحيحة',
        errors: errors.array()
      });
    }

    const contract = await Contract.findByPk(req.params.id);
    if (!contract) {
      return res.status(404).json({
        success: false,
        message: 'العقد غير موجود'
      });
    }

    // Validate dates if provided
    const startsFrom = req.body.startsFrom || contract.startsFrom;
    const endsAt = req.body.endsAt || contract.endsAt;
    
    if (new Date(endsAt) <= new Date(startsFrom)) {
      return res.status(400).json({
        success: false,
        message: 'تاريخ انتهاء العقد يجب أن يكون بعد تاريخ البداية'
      });
    }

    // Process new attachments
    let attachments = contract.attachments || [];
    if (req.files && req.files.length > 0) {
      for (const file of req.files) {
        attachments.push({
          name: file.originalname,
          attachment: file.filename,
          dateOfUpload: new Date(),
          uploadedBy: req.user.id
        });
      }
    }

    const updateData = {
      ...req.body,
      attachments,
      updatedBy: req.user.id
    };

    await contract.update(updateData);

    res.json({
      success: true,
      message: 'تم تحديث العقد بنجاح',
      data: contract
    });
  } catch (error) {
    console.error('Update contract error:', error);
    res.status(500).json({
      success: false,
      message: 'خطأ في تحديث العقد'
    });
  }
});

// Delete contract
router.delete('/:id', [
  authenticateToken,
  checkPermission('contracts', 'delete')
], async (req, res) => {
  try {
    const contract = await Contract.findByPk(req.params.id);
    if (!contract) {
      return res.status(404).json({
        success: false,
        message: 'العقد غير موجود'
      });
    }

    // Delete associated files
    if (contract.attachments && contract.attachments.length > 0) {
      for (const attachment of contract.attachments) {
        try {
          const filePath = path.join(__dirname, '../uploads/contracts', attachment.attachment);
          await fs.unlink(filePath);
        } catch (error) {
          console.error('Error deleting file:', error);
        }
      }
    }

    await contract.destroy();

    res.json({
      success: true,
      message: 'تم حذف العقد بنجاح'
    });
  } catch (error) {
    console.error('Delete contract error:', error);
    res.status(500).json({
      success: false,
      message: 'خطأ في حذف العقد'
    });
  }
});

// Delete attachment
router.delete('/:id/attachments/:attachmentIndex', [
  authenticateToken,
  checkPermission('contracts', 'edit')
], async (req, res) => {
  try {
    const contract = await Contract.findByPk(req.params.id);
    if (!contract) {
      return res.status(404).json({
        success: false,
        message: 'العقد غير موجود'
      });
    }

    const attachmentIndex = parseInt(req.params.attachmentIndex);
    const attachments = contract.attachments || [];

    if (attachmentIndex < 0 || attachmentIndex >= attachments.length) {
      return res.status(404).json({
        success: false,
        message: 'المرفق غير موجود'
      });
    }

    const attachment = attachments[attachmentIndex];

    // Delete file from filesystem
    try {
      const filePath = path.join(__dirname, '../uploads/contracts', attachment.attachment);
      await fs.unlink(filePath);
    } catch (error) {
      console.error('Error deleting file:', error);
    }

    // Remove attachment from array
    attachments.splice(attachmentIndex, 1);

    await contract.update({
      attachments,
      updatedBy: req.user.id
    });

    res.json({
      success: true,
      message: 'تم حذف المرفق بنجاح'
    });
  } catch (error) {
    console.error('Delete attachment error:', error);
    res.status(500).json({
      success: false,
      message: 'خطأ في حذف المرفق'
    });
  }
});

module.exports = router;
