const express = require('express');
const { body, validationResult, query } = require('express-validator');
const { Scene, Contract, User } = require('../models');
const { authenticateToken, checkPermission } = require('../middleware/auth');
const { Op } = require('sequelize');
const multer = require('multer');
const path = require('path');
const fs = require('fs').promises;

const router = express.Router();

// Configure multer for file uploads
const storage = multer.diskStorage({
  destination: async (req, file, cb) => {
    const uploadPath = path.join(__dirname, '../uploads/scenes');
    try {
      await fs.mkdir(uploadPath, { recursive: true });
      cb(null, uploadPath);
    } catch (error) {
      cb(error);
    }
  },
  filename: (req, file, cb) => {
    const uniqueSuffix = Date.now() + '-' + Math.round(Math.random() * 1E9);
    cb(null, `scene-${uniqueSuffix}${path.extname(file.originalname)}`);
  }
});

const upload = multer({
  storage,
  limits: {
    fileSize: parseInt(process.env.MAX_FILE_SIZE) || 10485760 // 10MB
  },
  fileFilter: (req, file, cb) => {
    const allowedTypes = /jpeg|jpg|png|gif|pdf|doc|docx|xls|xlsx/;
    const extname = allowedTypes.test(path.extname(file.originalname).toLowerCase());
    const mimetype = allowedTypes.test(file.mimetype);

    if (mimetype && extname) {
      return cb(null, true);
    } else {
      cb(new Error('نوع الملف غير مدعوم'));
    }
  }
});

// Get all scenes
router.get('/', [
  authenticateToken,
  checkPermission('scenes', 'view'),
  query('page').optional().isInt({ min: 1 }),
  query('limit').optional().isInt({ min: 1, max: 100 }),
  query('search').optional().isString(),
  query('contractId').optional().isInt(),
  query('action').optional().isString(),
  query('status').optional().isIn(['pending', 'in_progress', 'completed', 'cancelled']),
  query('dateFrom').optional().isISO8601(),
  query('dateTo').optional().isISO8601()
], async (req, res) => {
  try {
    const errors = validationResult(req);
    if (!errors.isEmpty()) {
      return res.status(400).json({
        success: false,
        message: 'بيانات غير صحيحة',
        errors: errors.array()
      });
    }

    const page = parseInt(req.query.page) || 1;
    const limit = parseInt(req.query.limit) || 10;
    const search = req.query.search || '';
    const contractId = req.query.contractId;
    const action = req.query.action;
    const status = req.query.status;
    const dateFrom = req.query.dateFrom;
    const dateTo = req.query.dateTo;
    const offset = (page - 1) * limit;

    let whereClause = {};

    // Search filter
    if (search) {
      whereClause[Op.or] = [
        { applicantName: { [Op.like]: `%${search}%` } },
        { applicantPhone: { [Op.like]: `%${search}%` } },
        { notes: { [Op.like]: `%${search}%` } },
        { facilityName: { [Op.like]: `%${search}%` } },
        { facilityOwnerName: { [Op.like]: `%${search}%` } },
        { ownerNationalID: { [Op.like]: `%${search}%` } },
        { facilityLocation: { [Op.like]: `%${search}%` } }
      ];
    }

    // Contract filter
    if (contractId) {
      whereClause.contractId = contractId;
    }

    // Action filter
    if (action) {
      whereClause.action = action;
    }

    // Status filter
    if (status) {
      whereClause.status = status;
    }

    // Date range filter
    if (dateFrom && dateTo) {
      whereClause.date = {
        [Op.between]: [dateFrom, dateTo]
      };
    } else if (dateFrom) {
      whereClause.date = {
        [Op.gte]: dateFrom
      };
    } else if (dateTo) {
      whereClause.date = {
        [Op.lte]: dateTo
      };
    }

    const { count, rows } = await Scene.findAndCountAll({
      where: whereClause,
      include: [
        {
          model: Contract,
          as: 'contract',
          attributes: ['id', 'contractNumber', 'secondParty', 'location'],
          required: false
        },
        {
          model: User,
          as: 'creator',
          attributes: ['id', 'fullName']
        }
      ],
      limit,
      offset,
      order: [['date', 'DESC'], ['createdAt', 'DESC']]
    });

    res.json({
      success: true,
      data: {
        scenes: rows,
        pagination: {
          currentPage: page,
          totalPages: Math.ceil(count / limit),
          totalItems: count,
          itemsPerPage: limit
        }
      }
    });
  } catch (error) {
    console.error('Get scenes error:', error);
    res.status(500).json({
      success: false,
      message: 'خطأ في جلب المشاهد'
    });
  }
});

// Get scene by ID
router.get('/:id', [
  authenticateToken,
  checkPermission('scenes', 'view')
], async (req, res) => {
  try {
    const scene = await Scene.findByPk(req.params.id, {
      include: [
        {
          model: Contract,
          as: 'contract',
          attributes: ['id', 'contractNumber', 'secondParty', 'location', 'ownerName', 'ownerPhoneNumber'],
          required: false
        },
        {
          model: User,
          as: 'creator',
          attributes: ['id', 'fullName']
        },
        {
          model: User,
          as: 'updater',
          attributes: ['id', 'fullName']
        }
      ]
    });

    if (!scene) {
      return res.status(404).json({
        success: false,
        message: 'المشهد غير موجود'
      });
    }

    res.json({
      success: true,
      data: scene
    });
  } catch (error) {
    console.error('Get scene error:', error);
    res.status(500).json({
      success: false,
      message: 'خطأ في جلب المشهد'
    });
  }
});

// Create new scene
router.post('/', [
  authenticateToken,
  checkPermission('scenes', 'add'),
  upload.array('attachments', 10),
  body('contractId').optional().isInt().withMessage('معرف العقد غير صحيح'),
  body('facilityName').isLength({ min: 2, max: 200 }).withMessage('اسم المنشأة مطلوب'),
  body('facilityOwnerName').isLength({ min: 2, max: 100 }).withMessage('اسم مالك المنشأة مطلوب'),
  body('ownerNationalID').isLength({ min: 10, max: 20 }).isNumeric().withMessage('رقم الهوية الوطنية غير صحيح'),
  body('facilityLocation').isLength({ min: 1 }).withMessage('موقع المنشأة مطلوب'),
  body('internalCameras').isInt({ min: 0 }).withMessage('عدد الكاميرات الداخلية غير صحيح'),
  body('externalCameras').isInt({ min: 0 }).withMessage('عدد الكاميرات الخارجية غير صحيح'),
  body('date').isISO8601().withMessage('التاريخ غير صحيح'),
  body('action').isIn(['صيانة', 'تركيب', 'فحص', 'إصلاح', 'استبدال', 'تحديث', 'أخرى']).withMessage('نوع العمل غير صحيح'),
  body('applicantName').isLength({ min: 2, max: 100 }).withMessage('اسم مقدم الطلب مطلوب'),
  body('applicantPhone').isLength({ min: 10, max: 20 }).isNumeric().withMessage('رقم هاتف مقدم الطلب غير صحيح'),
  body('status').optional().isIn(['pending', 'in_progress', 'completed', 'cancelled']),
  // body('materialsUsed').optional().isArray()
], async (req, res) => {
  try {
    const errors = validationResult(req);
    if (!errors.isEmpty()) {
      return res.status(400).json({
        success: false,
        message: 'بيانات غير صحيحة',
        errors: errors.array()
      });
    }

    // Check if contract exists (if contractId is provided)
    let contract = null;
    if (req.body.contractId) {
      contract = await Contract.findByPk(req.body.contractId);
      if (!contract) {
        return res.status(404).json({
          success: false,
          message: 'العقد غير موجود'
        });
      }
    }


    // Process attachments
    const attachments = [];
    if (req.files && req.files.length > 0) {
      for (const file of req.files) {
        attachments.push({
          name: file.originalname,
          attachment: file.filename,
          dateOfUpload: new Date(),
          uploadedBy: req.user.id
        });
      }
    }

    // If contract is provided, use its data; otherwise use provided facility data
    const sceneData = {
      ...req.body,
      attachments,
      createdBy: req.user.id
    };

    // If contract is provided, populate facility fields from contract
    if (contract) {
      sceneData.facilityName = contract.secondParty;
      sceneData.facilityOwnerName = contract.ownerName;
      sceneData.ownerNationalID = contract.ownerNationalID;
      sceneData.facilityLocation = contract.location;
      sceneData.internalCameras = contract.innerCamerasNumber;
      sceneData.externalCameras = contract.outerCamerasNumber;
    }

    const scene = await Scene.create(sceneData);

    // Include contract data in response
    const sceneWithContract = await Scene.findByPk(scene.id, {
      include: [
        {
          model: Contract,
          as: 'contract',
          attributes: ['id', 'contractNumber', 'secondParty', 'location'],
          required: false
        }
      ]
    });

    res.status(201).json({
      success: true,
      message: 'تم إنشاء المشهد بنجاح',
      data: sceneWithContract
    });
  } catch (error) {
    console.error('Create scene error:', error);
    res.status(500).json({
      success: false,
      message: 'خطأ في إنشاء المشهد'
    });
  }
});

// Update scene
router.put('/:id', [
  authenticateToken,
  checkPermission('scenes', 'edit'),
  upload.array('newAttachments', 10),
  body('contractId').optional().isInt().withMessage('معرف العقد غير صحيح'),
  body('facilityName').optional().isLength({ min: 2, max: 200 }).withMessage('اسم المنشأة غير صحيح'),
  body('facilityOwnerName').optional().isLength({ min: 2, max: 100 }).withMessage('اسم مالك المنشأة غير صحيح'),
  body('ownerNationalID').optional().isLength({ min: 10, max: 20 }).isNumeric().withMessage('رقم الهوية الوطنية غير صحيح'),
  body('facilityLocation').optional().isLength({ min: 1 }).withMessage('موقع المنشأة غير صحيح'),
  body('internalCameras').optional().isInt({ min: 0 }).withMessage('عدد الكاميرات الداخلية غير صحيح'),
  body('externalCameras').optional().isInt({ min: 0 }).withMessage('عدد الكاميرات الخارجية غير صحيح'),
  body('date').optional().isISO8601(),
  body('action').optional().isIn(['صيانة', 'تركيب', 'فحص', 'إصلاح', 'استبدال', 'تحديث', 'أخرى']),
  body('applicantName').optional().isLength({ min: 2, max: 100 }),
  body('applicantPhone').optional().isLength({ min: 10, max: 20 }).isNumeric(),
  body('status').optional().isIn(['pending', 'in_progress', 'completed', 'cancelled']),
  // body('materialsUsed').optional().isArray()
], async (req, res) => {
  try {
    const errors = validationResult(req);
    if (!errors.isEmpty()) {
      return res.status(400).json({
        success: false,
        message: 'بيانات غير صحيحة',
        errors: errors.array()
      });
    }

    const scene = await Scene.findByPk(req.params.id);
    if (!scene) {
      return res.status(404).json({
        success: false,
        message: 'المشهد غير موجود'
      });
    }


    // Process new attachments
    let attachments = scene.attachments || [];
    if (req.files && req.files.length > 0) {
      for (const file of req.files) {
        attachments.push({
          name: file.originalname,
          attachment: file.filename,
          dateOfUpload: new Date(),
          uploadedBy: req.user.id
        });
      }
    }

    const updateData = {
      ...req.body,
      attachments,
      updatedBy: req.user.id
    };

    await scene.update(updateData);

    // Include contract data in response
    const updatedScene = await Scene.findByPk(scene.id, {
      include: [
        {
          model: Contract,
          as: 'contract',
          attributes: ['id', 'contractNumber', 'secondParty', 'location'],
          required: false
        }
      ]
    });

    res.json({
      success: true,
      message: 'تم تحديث المشهد بنجاح',
      data: updatedScene
    });
  } catch (error) {
    console.error('Update scene error:', error);
    res.status(500).json({
      success: false,
      message: 'خطأ في تحديث المشهد'
    });
  }
});

// Delete scene
router.delete('/:id', [
  authenticateToken,
  checkPermission('scenes', 'delete')
], async (req, res) => {
  try {
    const scene = await Scene.findByPk(req.params.id);
    if (!scene) {
      return res.status(404).json({
        success: false,
        message: 'المشهد غير موجود'
      });
    }

    // Delete associated files
    if (scene.attachments && scene.attachments.length > 0) {
      for (const attachment of scene.attachments) {
        try {
          const filePath = path.join(__dirname, '../uploads/scenes', attachment.attachment);
          await fs.unlink(filePath);
        } catch (error) {
          console.error('Error deleting file:', error);
        }
      }
    }

    await scene.destroy();

    res.json({
      success: true,
      message: 'تم حذف المشهد بنجاح'
    });
  } catch (error) {
    console.error('Delete scene error:', error);
    res.status(500).json({
      success: false,
      message: 'خطأ في حذف المشهد'
    });
  }
});

// Delete attachment
router.delete('/:id/attachments/:attachmentIndex', [
  authenticateToken,
  checkPermission('scenes', 'edit')
], async (req, res) => {
  try {
    const scene = await Scene.findByPk(req.params.id);
    if (!scene) {
      return res.status(404).json({
        success: false,
        message: 'المشهد غير موجود'
      });
    }

    const attachmentIndex = parseInt(req.params.attachmentIndex);
    const attachments = scene.attachments || [];

    if (attachmentIndex < 0 || attachmentIndex >= attachments.length) {
      return res.status(404).json({
        success: false,
        message: 'المرفق غير موجود'
      });
    }

    const attachment = attachments[attachmentIndex];

    // Delete file from filesystem
    try {
      const filePath = path.join(__dirname, '../uploads/scenes', attachment.attachment);
      await fs.unlink(filePath);
    } catch (error) {
      console.error('Error deleting file:', error);
    }

    // Remove attachment from array
    attachments.splice(attachmentIndex, 1);

    await scene.update({
      attachments,
      updatedBy: req.user.id
    });

    res.json({
      success: true,
      message: 'تم حذف المرفق بنجاح'
    });
  } catch (error) {
    console.error('Delete attachment error:', error);
    res.status(500).json({
      success: false,
      message: 'خطأ في حذف المرفق'
    });
  }
});

// Get scenes by contract
router.get('/contract/:contractId', [
  authenticateToken,
  checkPermission('scenes', 'view')
], async (req, res) => {
  try {
    const scenes = await Scene.findAll({
      where: { contractId: req.params.contractId },
      include: [
        {
          model: User,
          as: 'creator',
          attributes: ['id', 'fullName']
        }
      ],
      order: [['date', 'DESC'], ['createdAt', 'DESC']]
    });

    res.json({
      success: true,
      data: scenes
    });
  } catch (error) {
    console.error('Get scenes by contract error:', error);
    res.status(500).json({
      success: false,
      message: 'خطأ في جلب مشاهد العقد'
    });
  }
});

// Get contract facility data for scene creation
router.get('/contract-data/:contractId', [
  authenticateToken,
  checkPermission('scenes', 'view')
], async (req, res) => {
  try {
    const contract = await Contract.findByPk(req.params.contractId, {
      attributes: [
        'id', 
        'contractNumber', 
        'secondParty', 
        'ownerName', 
        'ownerNationalID', 
        'location', 
        'innerCamerasNumber', 
        'outerCamerasNumber'
      ]
    });

    if (!contract) {
      return res.status(404).json({
        success: false,
        message: 'العقد غير موجود'
      });
    }

    // Map contract fields to scene facility fields
    const facilityData = {
      facilityName: contract.secondParty,
      facilityOwnerName: contract.ownerName,
      ownerNationalID: contract.ownerNationalID,
      facilityLocation: contract.location,
      internalCameras: contract.innerCamerasNumber,
      externalCameras: contract.outerCamerasNumber
    };

    res.json({
      success: true,
      data: facilityData
    });
  } catch (error) {
    console.error('Get contract facility data error:', error);
    res.status(500).json({
      success: false,
      message: 'خطأ في جلب بيانات العقد'
    });
  }
});

module.exports = router;
