const express = require('express');
const { body, validationResult } = require('express-validator');
const { Settings } = require('../models');
const { authenticateToken, requireAdmin } = require('../middleware/auth');

const router = express.Router();

// Get all settings
router.get('/', [
  authenticateToken,
  requireAdmin
], async (req, res) => {
  try {
    const settings = await Settings.findAll({
      order: [['category', 'ASC'], ['key', 'ASC']]
    });

    // Group settings by category
    const groupedSettings = settings.reduce((acc, setting) => {
      if (!acc[setting.category]) {
        acc[setting.category] = [];
      }
      
      let value = setting.value;
      switch (setting.type) {
        case 'number':
          value = parseFloat(value);
          break;
        case 'boolean':
          value = value === 'true';
          break;
        case 'json':
          try {
            value = JSON.parse(value);
          } catch (e) {
            value = setting.value;
          }
          break;
      }
      
      acc[setting.category].push({
        id: setting.id,
        key: setting.key,
        value: value,
        type: setting.type,
        description: setting.description,
        updatedAt: setting.updatedAt
      });
      
      return acc;
    }, {});

    res.json({
      success: true,
      data: groupedSettings
    });
  } catch (error) {
    console.error('Get settings error:', error);
    res.status(500).json({
      success: false,
      message: 'خطأ في جلب الإعدادات'
    });
  }
});

// Get setting by key
router.get('/:key', [
  authenticateToken,
  requireAdmin
], async (req, res) => {
  try {
    const setting = await Settings.findOne({
      where: { key: req.params.key }
    });

    if (!setting) {
      return res.status(404).json({
        success: false,
        message: 'الإعداد غير موجود'
      });
    }

    let value = setting.value;
    switch (setting.type) {
      case 'number':
        value = parseFloat(value);
        break;
      case 'boolean':
        value = value === 'true';
        break;
      case 'json':
        try {
          value = JSON.parse(value);
        } catch (e) {
          value = setting.value;
        }
        break;
    }

    res.json({
      success: true,
      data: {
        key: setting.key,
        value: value,
        type: setting.type,
        description: setting.description
      }
    });
  } catch (error) {
    console.error('Get setting error:', error);
    res.status(500).json({
      success: false,
      message: 'خطأ في جلب الإعداد'
    });
  }
});

// Update or create setting
router.put('/:key', [
  authenticateToken,
  requireAdmin,
  body('value').notEmpty().withMessage('القيمة مطلوبة'),
  body('type').optional().isIn(['string', 'number', 'boolean', 'json']),
  body('description').optional().isString(),
  body('category').optional().isString()
], async (req, res) => {
  try {
    const errors = validationResult(req);
    if (!errors.isEmpty()) {
      return res.status(400).json({
        success: false,
        message: 'بيانات غير صحيحة',
        errors: errors.array()
      });
    }

    const { value, type = 'string', description, category = 'general' } = req.body;
    const key = req.params.key;

    const setting = await Settings.setValue(key, value, type, req.user.id);

    if (description) {
      await setting.update({ description });
    }

    if (category) {
      await setting.update({ category });
    }

    res.json({
      success: true,
      message: 'تم تحديث الإعداد بنجاح',
      data: {
        key: setting.key,
        value: value,
        type: setting.type,
        description: setting.description,
        category: setting.category
      }
    });
  } catch (error) {
    console.error('Update setting error:', error);
    res.status(500).json({
      success: false,
      message: 'خطأ في تحديث الإعداد'
    });
  }
});

// Delete setting
router.delete('/:key', [
  authenticateToken,
  requireAdmin
], async (req, res) => {
  try {
    const setting = await Settings.findOne({
      where: { key: req.params.key }
    });

    if (!setting) {
      return res.status(404).json({
        success: false,
        message: 'الإعداد غير موجود'
      });
    }

    await setting.destroy();

    res.json({
      success: true,
      message: 'تم حذف الإعداد بنجاح'
    });
  } catch (error) {
    console.error('Delete setting error:', error);
    res.status(500).json({
      success: false,
      message: 'خطأ في حذف الإعداد'
    });
  }
});

// Initialize default settings
router.post('/initialize', [
  authenticateToken,
  requireAdmin
], async (req, res) => {
  try {
    const defaultSettings = [
      // Contract settings
      {
        key: 'next_contract_number',
        value: '1',
        type: 'number',
        description: 'رقم العقد التالي',
        category: 'contracts'
      },
      {
        key: 'contract_number_prefix',
        value: 'C-',
        type: 'string',
        description: 'بادئة رقم العقد',
        category: 'contracts'
      },
      
      // Print settings
      {
        key: 'print_header',
        value: 'نظام إدارة عقود الصيانة',
        type: 'string',
        description: 'رأس الصفحة للطباعة',
        category: 'print'
      },
      {
        key: 'print_footer',
        value: 'جميع الحقوق محفوظة',
        type: 'string',
        description: 'تذييل الصفحة للطباعة',
        category: 'print'
      },
      {
        key: 'print_margins',
        value: JSON.stringify({
          top: '20mm',
          bottom: '20mm',
          left: '15mm',
          right: '15mm'
        }),
        type: 'json',
        description: 'هوامش الصفحة للطباعة',
        category: 'print'
      },
      
      // Backup settings
      {
        key: 'backup_enabled',
        value: 'true',
        type: 'boolean',
        description: 'تفعيل النسخ الاحتياطي التلقائي',
        category: 'backup'
      },
      {
        key: 'backup_schedule',
        value: '0 2 * * *',
        type: 'string',
        description: 'جدولة النسخ الاحتياطي (cron format)',
        category: 'backup'
      },
      {
        key: 'backup_retention_days',
        value: '30',
        type: 'number',
        description: 'عدد أيام الاحتفاظ بالنسخ الاحتياطية',
        category: 'backup'
      },
      
      // System settings
      {
        key: 'system_name',
        value: 'نظام إدارة عقود الصيانة',
        type: 'string',
        description: 'اسم النظام',
        category: 'system'
      },
      {
        key: 'max_file_size',
        value: '10485760',
        type: 'number',
        description: 'الحد الأقصى لحجم الملف (بايت)',
        category: 'system'
      },
      {
        key: 'allowed_file_types',
        value: JSON.stringify(['jpg', 'jpeg', 'png', 'gif', 'pdf', 'doc', 'docx', 'xls', 'xlsx']),
        type: 'json',
        description: 'أنواع الملفات المسموحة',
        category: 'system'
      }
    ];

    for (const settingData of defaultSettings) {
      await Settings.findOrCreate({
        where: { key: settingData.key },
        defaults: {
          ...settingData,
          updatedBy: req.user.id
        }
      });
    }

    res.json({
      success: true,
      message: 'تم تهيئة الإعدادات الافتراضية بنجاح'
    });
  } catch (error) {
    console.error('Initialize settings error:', error);
    res.status(500).json({
      success: false,
      message: 'خطأ في تهيئة الإعدادات'
    });
  }
});

// Bulk update settings
router.put('/', [
  authenticateToken,
  requireAdmin,
  body('settings').isArray().withMessage('الإعدادات يجب أن تكون مصفوفة')
], async (req, res) => {
  try {
    const errors = validationResult(req);
    if (!errors.isEmpty()) {
      return res.status(400).json({
        success: false,
        message: 'بيانات غير صحيحة',
        errors: errors.array()
      });
    }

    const { settings } = req.body;
    const updatedSettings = [];

    for (const settingData of settings) {
      if (!settingData.key || settingData.value === undefined) {
        continue;
      }

      const setting = await Settings.setValue(
        settingData.key,
        settingData.value,
        settingData.type || 'string',
        req.user.id
      );

      if (settingData.description) {
        await setting.update({ description: settingData.description });
      }

      if (settingData.category) {
        await setting.update({ category: settingData.category });
      }

      updatedSettings.push({
        key: setting.key,
        value: settingData.value,
        type: setting.type
      });
    }

    res.json({
      success: true,
      message: 'تم تحديث الإعدادات بنجاح',
      data: updatedSettings
    });
  } catch (error) {
    console.error('Bulk update settings error:', error);
    res.status(500).json({
      success: false,
      message: 'خطأ في تحديث الإعدادات'
    });
  }
});

module.exports = router;
