const jwt = require('jsonwebtoken');
const { User } = require('../models');

const authenticateToken = async (req, res, next) => {
  try {
    const authHeader = req.headers['authorization'];
    const token = authHeader && authHeader.split(' ')[1];

    if (!token) {
      return res.status(401).json({
        success: false,
        message: 'رمز الوصول مطلوب'
      });
    }

    const decoded = jwt.verify(token, process.env.JWT_SECRET);
    const user = await User.findByPk(decoded.userId);

    if (!user || !user.isActive) {
      return res.status(401).json({
        success: false,
        message: 'المستخدم غير موجود أو غير مفعل'
      });
    }

    req.user = user;
    next();
  } catch (error) {
    console.error('Authentication error:', error);
    return res.status(403).json({
      success: false,
      message: 'رمز الوصول غير صحيح'
    });
  }
};

const requireAdmin = (req, res, next) => {
  if (req.user.userType !== 'admin') {
    return res.status(403).json({
      success: false,
      message: 'هذه العملية تتطلب صلاحيات المدير'
    });
  }
  next();
};

const checkPermission = (resource, action) => {
  return (req, res, next) => {
    const userPermissions = req.user.permissions;
    
    if (req.user.userType === 'admin') {
      return next();
    }
    if (!JSON.parse(userPermissions)?.[resource]?.[action]) {
      return res.status(403).json({
        success: false,
        message: 'ليس لديك صلاحية للقيام بهذه العملية'
      });
    }

    next();
  };
};

module.exports = {
  authenticateToken,
  requireAdmin,
  checkPermission
};
