const { DataTypes } = require('sequelize');
const { sequelize } = require('../config/database');

const Settings = sequelize.define('Settings', {
  id: {
    type: DataTypes.INTEGER,
    primaryKey: true,
    autoIncrement: true
  },
  key: {
    type: DataTypes.STRING(100),
    allowNull: false,
    unique: true
  },
  value: {
    type: DataTypes.TEXT,
    allowNull: false
  },
  type: {
    type: DataTypes.ENUM('string', 'number', 'boolean', 'json'),
    defaultValue: 'string'
  },
  description: {
    type: DataTypes.TEXT,
    allowNull: true
  },
  category: {
    type: DataTypes.STRING(50),
    allowNull: false,
    defaultValue: 'general'
  },
  updatedBy: {
    type: DataTypes.INTEGER,
    allowNull: true,
    references: {
      model: 'users',
      key: 'id'
    }
  }
}, {
  tableName: 'settings',
  timestamps: true,
  indexes: [
    {
      fields: ['key']
    },
    {
      fields: ['category']
    }
  ]
});

// Static method to get setting value
Settings.getValue = async function(key, defaultValue = null) {
  try {
    const setting = await this.findOne({ where: { key } });
    if (!setting) return defaultValue;
    
    switch (setting.type) {
      case 'number':
        return parseFloat(setting.value);
      case 'boolean':
        return setting.value === 'true';
      case 'json':
        return JSON.parse(setting.value);
      default:
        return setting.value;
    }
  } catch (error) {
    console.error('Error getting setting:', error);
    return defaultValue;
  }
};

// Static method to set setting value
Settings.setValue = async function(key, value, type = 'string', userId = null) {
  try {
    let stringValue;
    switch (type) {
      case 'json':
        stringValue = JSON.stringify(value);
        break;
      case 'boolean':
        stringValue = value ? 'true' : 'false';
        break;
      default:
        stringValue = String(value);
    }
    
    const [setting, created] = await this.findOrCreate({
      where: { key },
      defaults: {
        key,
        value: stringValue,
        type,
        updatedBy: userId
      }
    });
    
    if (!created) {
      await setting.update({
        value: stringValue,
        type,
        updatedBy: userId
      });
    }
    
    return setting;
  } catch (error) {
    console.error('Error setting value:', error);
    throw error;
  }
};

module.exports = Settings;
