// API Handler for Backend Communications

class ApiClient {
    constructor() {
        this.baseURL = '/api';
        this.token = localStorage.getItem('authToken');
    }

    // Set authentication token
    setToken(token) {
        this.token = token;
        if (token) {
            localStorage.setItem('authToken', token);
        } else {
            localStorage.removeItem('authToken');
        }
    }

    // Get authentication headers
    getHeaders(contentType = 'application/json') {
        const headers = {};
        
        if (contentType) {
            headers['Content-Type'] = contentType;
        }
        
        if (this.token) {
            headers['Authorization'] = `Bearer ${this.token}`;
        }
        
        return headers;
    }

    // Generic request method
    async request(endpoint, options = {}) {
        const url = `${this.baseURL}${endpoint}`;
        const config = {
            headers: this.getHeaders(options.contentType),
            ...options
        };

        try {
            const response = await fetch(url, config);
            
            // Handle authentication errors
            if (response.status === 401) {
                this.setToken(null);
                window.location.reload();
                return;
            }

            const data = await response.json();
            
            if (!response.ok) {
                throw new Error(data.message || 'حدث خطأ في الخادم');
            }

            return data;
        } catch (error) {
            console.error('API Error:', error);
            throw error;
        }
    }

    // GET request
    async get(endpoint, params = {}) {
        const queryString = new URLSearchParams(params).toString();
        const url = queryString ? `${endpoint}?${queryString}` : endpoint;
        
        return this.request(url, {
            method: 'GET'
        });
    }

    // POST request
    async post(endpoint, data = {}) {
        return this.request(endpoint, {
            method: 'POST',
            body: JSON.stringify(data)
        });
    }

    // PUT request
    async put(endpoint, data = {}) {
        return this.request(endpoint, {
            method: 'PUT',
            body: JSON.stringify(data)
        });
    }

    // DELETE request
    async delete(endpoint) {
        return this.request(endpoint, {
            method: 'DELETE'
        });
    }

    // POST with FormData (for file uploads)
    async postFormData(endpoint, formData) {
        return this.request(endpoint, {
            method: 'POST',
            body: formData,
            contentType: null // Let browser set content-type for FormData
        });
    }

    // PUT with FormData (for file uploads)
    async putFormData(endpoint, formData) {
        return this.request(endpoint, {
            method: 'PUT',
            body: formData,
            contentType: null
        });
    }

    // Download file
    async downloadFile(endpoint, filename) {
        const url = `${this.baseURL}${endpoint}`;
        const response = await fetch(url, {
            headers: this.getHeaders(null)
        });

        if (!response.ok) {
            throw new Error('فشل في تحميل الملف');
        }

        const blob = await response.blob();
        const downloadUrl = window.URL.createObjectURL(blob);
        
        const link = document.createElement('a');
        link.href = downloadUrl;
        link.download = filename;
        document.body.appendChild(link);
        link.click();
        document.body.removeChild(link);
        
        window.URL.revokeObjectURL(downloadUrl);
    }

    // Authentication methods
    async login(credentials) {
        const response = await this.post('/auth/login', credentials);
        if (response.success && response.data.token) {
            this.setToken(response.data.token);
        }
        return response;
    }

    async logout() {
        try {
            await this.post('/auth/logout');
        } catch (error) {
            console.error('Logout error:', error);
        } finally {
            this.setToken(null);
        }
    }

    async getCurrentUser() {
        return this.get('/auth/me');
    }

    async changePassword(passwordData) {
        return this.put('/auth/change-password', passwordData);
    }

    // User management methods
    async getUsers(params = {}) {
        return this.get('/users', params);
    }

    async getUser(id) {
        return this.get(`/users/${id}`);
    }

    async createUser(userData) {
        return this.post('/users', userData);
    }

    async updateUser(id, userData) {
        return this.put(`/users/${id}`, userData);
    }

    async deleteUser(id) {
        return this.delete(`/users/${id}`);
    }

    async resetUserPassword(id, passwordData) {
        return this.put(`/users/${id}/reset-password`, passwordData);
    }

    // Contract methods
    async getContracts(params = {}) {
        return this.get('/contracts', params);
    }

    async getContract(id) {
        return this.get(`/contracts/${id}`);
    }

    async getNextContractNumber() {
        return this.get('/contracts/next-number');
    }

    async createContract(contractData, files = []) {
        const formData = new FormData();
        
        // Add contract data
        Object.keys(contractData).forEach(key => {
            if (contractData[key] !== null && contractData[key] !== undefined) {
                formData.append(key, contractData[key]);
            }
        });
        
        // Add files
        files.forEach(file => {
            formData.append('attachments', file);
        });
        
        return this.postFormData('/contracts', formData);
    }

    async updateContract(id, contractData, files = []) {
        const formData = new FormData();
        
        // Add contract data
        Object.keys(contractData).forEach(key => {
            if (contractData[key] !== null && contractData[key] !== undefined) {
                formData.append(key, contractData[key]);
            }
        });
        
        // Add new files
        files.forEach(file => {
            formData.append('newAttachments', file);
        });
        
        return this.putFormData(`/contracts/${id}`, formData);
    }

    async deleteContract(id) {
        return this.delete(`/contracts/${id}`);
    }

    async deleteContractAttachment(contractId, attachmentIndex) {
        return this.delete(`/contracts/${contractId}/attachments/${attachmentIndex}`);
    }

    // Scene methods
    async getScenes(params = {}) {
        return this.get('/scenes', params);
    }

    async getScene(id) {
        return this.get(`/scenes/${id}`);
    }

    async getScenesByContract(contractId) {
        return this.get(`/scenes/contract/${contractId}`);
    }

    async createScene(sceneData, files = []) {
        const formData = new FormData();
        
        // Add scene data
        Object.keys(sceneData).forEach(key => {
            if (sceneData[key] !== null && sceneData[key] !== undefined) {
                formData.append(key, sceneData[key]);
            }
        });
        
        // Add files
        files.forEach(file => {
            formData.append('attachments', file);
        });
        
        return this.postFormData('/scenes', formData);
    }

    async updateScene(id, sceneData, files = []) {
        const formData = new FormData();
        
        // Add scene data
        Object.keys(sceneData).forEach(key => {
            if (sceneData[key] !== null && sceneData[key] !== undefined) {
                formData.append(key, sceneData[key]);
            }
        });
        
        // Add new files
        files.forEach(file => {
            formData.append('newAttachments', file);
        });
        
        return this.putFormData(`/scenes/${id}`, formData);
    }

    async deleteScene(id) {
        return this.delete(`/scenes/${id}`);
    }

    async deleteSceneAttachment(sceneId, attachmentIndex) {
        return this.delete(`/scenes/${sceneId}/attachments/${attachmentIndex}`);
    }

    // Report methods
    async getDashboardStats() {
        return this.get('/reports/dashboard');
    }

    async downloadContractReport(contractId, options = {}) {
        const params = new URLSearchParams(options).toString();
        const endpoint = `/reports/contract/${contractId}${params ? '?' + params : ''}`;
        const filename = `contract-${contractId}-report.${options.format || 'pdf'}`;
        return this.downloadFile(endpoint, filename);
    }

    async downloadSceneReport(sceneId, options = {}) {
        const params = new URLSearchParams(options).toString();
        const endpoint = `/reports/scene/${sceneId}${params ? '?' + params : ''}`;
        const filename = `scene-${sceneId}-report.${options.format || 'pdf'}`;
        return this.downloadFile(endpoint, filename);
    }

    async downloadTimeRangeReport(options = {}) {
        const params = new URLSearchParams(options).toString();
        const endpoint = `/reports/time-range${params ? '?' + params : ''}`;
        const filename = `${options.type || 'report'}-${options.dateFrom}-${options.dateTo}.${options.format || 'pdf'}`;
        return this.downloadFile(endpoint, filename);
    }

    // Settings methods
    async getSettings() {
        return this.get('/settings');
    }

    async getSetting(key) {
        return this.get(`/settings/${key}`);
    }

    async updateSetting(key, settingData) {
        return this.put(`/settings/${key}`, settingData);
    }

    async deleteSetting(key) {
        return this.delete(`/settings/${key}`);
    }

    async initializeSettings() {
        return this.post('/settings/initialize');
    }

    async bulkUpdateSettings(settings) {
        return this.put('/settings', { settings });
    }

    // Backup methods
    async getBackups() {
        return this.get('/backup/list');
    }

    async createBackup() {
        return this.post('/backup/create');
    }

    async downloadBackup(filename) {
        return this.downloadFile(`/backup/download/${filename}`, filename);
    }

    async deleteBackup(filename) {
        return this.delete(`/backup/${filename}`);
    }

    async restoreBackup(filename) {
        return this.post(`/backup/restore/${filename}`, { confirmRestore: true });
    }

    async getBackupSettings() {
        return this.get('/backup/settings');
    }

    async updateBackupSettings(settings) {
        return this.put('/backup/settings', settings);
    }

    async cleanOldBackups() {
        return this.post('/backup/clean');
    }
}

// Create global API instance
const api = new ApiClient();

// Export for use in other modules
window.api = api;
