// Authentication Handler

class AuthManager {
    constructor() {
        this.loginModal = null;
        this.changePasswordModal = null;
        this.init();
    }

    init() {
        this.setupLoginModal();
        this.setupChangePasswordModal();
        this.setupEventListeners();
        this.checkAuthStatus();
    }

    setupLoginModal() {
        this.loginModal = new bootstrap.Modal(document.getElementById('loginModal'));
        
        // Handle login form submission
        const loginForm = document.getElementById('loginForm');
        loginForm.addEventListener('submit', (e) => {
            e.preventDefault();
            this.handleLogin();
        });

        // Toggle password visibility
        const togglePassword = document.getElementById('togglePassword');
        const passwordInput = document.getElementById('password');
        
        togglePassword.addEventListener('click', () => {
            const type = passwordInput.getAttribute('type') === 'password' ? 'text' : 'password';
            passwordInput.setAttribute('type', type);
            
            const icon = togglePassword.querySelector('i');
            icon.classList.toggle('fa-eye');
            icon.classList.toggle('fa-eye-slash');
        });
    }

    setupChangePasswordModal() {
        this.changePasswordModal = new bootstrap.Modal(document.getElementById('changePasswordModal'));
        
        // Handle change password form
        const savePasswordBtn = document.getElementById('savePasswordBtn');
        savePasswordBtn.addEventListener('click', () => {
            this.handleChangePassword();
        });

        // Validate password confirmation
        const newPassword = document.getElementById('newPassword');
        const confirmPassword = document.getElementById('confirmPassword');
        
        confirmPassword.addEventListener('input', () => {
            if (newPassword.value !== confirmPassword.value) {
                confirmPassword.setCustomValidity('كلمات المرور غير متطابقة');
            } else {
                confirmPassword.setCustomValidity('');
            }
        });
    }

    setupEventListeners() {
        // Logout button
        const logoutBtn = document.getElementById('logoutBtn');
        logoutBtn.addEventListener('click', (e) => {
            e.preventDefault();
            this.handleLogout();
        });

        // Change password button
        const changePasswordBtn = document.getElementById('changePasswordBtn');
        changePasswordBtn.addEventListener('click', (e) => {
            e.preventDefault();
            this.showChangePasswordModal();
        });
    }

    async checkAuthStatus() {
        const token = localStorage.getItem('authToken');
        
        if (!token) {
            this.showLoginModal();
            return;
        }

        try {
            utils.showLoading();
            const response = await api.getCurrentUser();
            
            if (response.success) {
                this.handleLoginSuccess(response.data, token);
            } else {
                this.showLoginModal();
            }
        } catch (error) {
            console.error('Auth check failed:', error);
            this.showLoginModal();
        } finally {
            utils.hideLoading();
        }
    }

    showLoginModal() {
        document.getElementById('mainApp').classList.add('d-none');
        this.loginModal.show();
    }

    hideLoginModal() {
        this.loginModal.hide();
        document.getElementById('mainApp').classList.remove('d-none');
    }

    async handleLogin() {
        const userName = document.getElementById('userName').value.trim();
        const password = document.getElementById('password').value;

        if (!userName || !password) {
            utils.showToast('يرجى إدخال اسم المستخدم وكلمة المرور', 'warning');
            return;
        }

        try {
            utils.showLoading();
            
            const response = await api.login({ userName, password });
            
            if (response.success) {
                this.handleLoginSuccess(response.data.user, response.data.token);
                utils.showToast('تم تسجيل الدخول بنجاح', 'success');
            } else {
                utils.showToast(response.message || 'فشل في تسجيل الدخول', 'danger');
            }
        } catch (error) {
            console.error('Login error:', error);
            utils.showToast(error.message || 'حدث خطأ أثناء تسجيل الدخول', 'danger');
        } finally {
            utils.hideLoading();
        }
    }

    handleLoginSuccess(user, token) {
        // Store user data and token
        utils.setCurrentUser(user);
        api.setToken(token);
        
        // Hide login modal and show main app
        this.hideLoginModal();
        
        // Reset login form
        document.getElementById('loginForm').reset();
        
        // Initialize the main application
        if (window.app && window.app.init) {
            window.app.init();
        }
    }

    async handleLogout() {
        utils.showConfirmDialog(
            'تسجيل الخروج',
            'هل أنت متأكد من رغبتك في تسجيل الخروج؟',
            async () => {
                try {
                    utils.showLoading();
                    await api.logout();
                } catch (error) {
                    console.error('Logout error:', error);
                } finally {
                    // Clear user data
                    utils.clearCurrentUser();
                    
                    // Reload page to reset application state
                    window.location.reload();
                }
            }
        );
    }

    showChangePasswordModal() {
        // Reset form
        const form = document.getElementById('changePasswordForm');
        form.reset();
        utils.clearFormValidation(form);
        
        this.changePasswordModal.show();
    }

    async handleChangePassword() {
        const form = document.getElementById('changePasswordForm');
        
        if (!utils.validateForm(form)) {
            utils.showToast('يرجى ملء جميع الحقول المطلوبة', 'warning');
            return;
        }

        const currentPassword = document.getElementById('currentPassword').value;
        const newPassword = document.getElementById('newPassword').value;
        const confirmPassword = document.getElementById('confirmPassword').value;

        if (newPassword !== confirmPassword) {
            utils.showToast('كلمات المرور الجديدة غير متطابقة', 'warning');
            return;
        }

        if (newPassword.length < 6) {
            utils.showToast('كلمة المرور الجديدة يجب أن تكون 6 أحرف على الأقل', 'warning');
            return;
        }

        try {
            utils.showLoading();
            
            const response = await api.changePassword({
                currentPassword,
                newPassword
            });
            
            if (response.success) {
                utils.showToast('تم تغيير كلمة المرور بنجاح', 'success');
                this.changePasswordModal.hide();
                form.reset();
            } else {
                utils.showToast(response.message || 'فشل في تغيير كلمة المرور', 'danger');
            }
        } catch (error) {
            console.error('Change password error:', error);
            utils.showToast(error.message || 'حدث خطأ أثناء تغيير كلمة المرور', 'danger');
        } finally {
            utils.hideLoading();
        }
    }

    // Check if user has permission
    hasPermission(resource, action) {
        const user = utils.getCurrentUser();
        if (!user) return false;
        
        // Admin has all permissions
        if (user.userType === 'admin') return true;
        
        // Check specific permission
        return user.permissions && 
               user.permissions[resource] && 
               user.permissions[resource][action];
    }

    // Get current user
    getCurrentUser() {
        return utils.getCurrentUser();
    }

    // Check if current user is admin
    isAdmin() {
        const user = this.getCurrentUser();
        return user && user.userType === 'admin';
    }

    // Require authentication
    requireAuth() {
        const token = localStorage.getItem('authToken');
        if (!token) {
            this.showLoginModal();
            return false;
        }
        return true;
    }

    // Require admin privileges
    requireAdmin() {
        if (!this.requireAuth()) return false;
        
        if (!this.isAdmin()) {
            utils.showToast('هذه العملية تتطلب صلاحيات المدير', 'warning');
            return false;
        }
        
        return true;
    }

    // Require specific permission
    requirePermission(resource, action) {
        if (!this.requireAuth()) return false;
        
        if (!this.hasPermission(resource, action)) {
            utils.showToast('ليس لديك صلاحية للقيام بهذه العملية', 'warning');
            return false;
        }
        
        return true;
    }
}

// Create global auth manager instance
const auth = new AuthManager();

// Export for use in other modules
window.auth = auth;
