const express = require('express');
const router = express.Router();
const path = require('path');
const fs = require('fs');
const { upload, handleUploadError, deleteFile, getFileInfo, contractsDir } = require('../middleware/upload');
const { authenticateToken } = require('../middleware/auth');
const Contract = require('../models/Contract');

// Upload files for a contract
router.post('/contracts/:contractId', authenticateToken, (req, res) => {
  upload.array('files', 10)(req, res, async (err) => {
    if (err) {
      return handleUploadError(err, req, res, () => {
        res.status(500).json({
          success: false,
          message: 'خطأ في رفع الملفات'
        });
      });
    }

    try {
      const { contractId } = req.params;
      const { description } = req.body;

      // Check if contract exists
      const contract = await Contract.findByPk(contractId);
      if (!contract) {
        // Delete uploaded files if contract doesn't exist
        if (req.files) {
          req.files.forEach(file => {
            deleteFile(file.path);
          });
        }
        return res.status(404).json({
          success: false,
          message: 'العقد غير موجود'
        });
      }

      if (!req.files || req.files.length === 0) {
        return res.status(400).json({
          success: false,
          message: 'لم يتم اختيار أي ملفات'
        });
      }

      // Process uploaded files
      const newAttachments = req.files.map(file => ({
        id: Date.now() + Math.random(),
        filename: file.filename,
        originalName: file.originalname,
        mimetype: file.mimetype,
        size: file.size,
        description: description || '',
        uploadedAt: new Date().toISOString(),
        uploadedBy: req.user.id
      }));

      // Get current attachments and add new ones
      const currentAttachments = contract.attachments || [];
      const updatedAttachments = [...currentAttachments, ...newAttachments];

      // Update contract with new attachments
      await contract.update({
        attachments: updatedAttachments,
        updatedBy: req.user.id
      });

      res.json({
        success: true,
        message: `تم رفع ${newAttachments.length} ملف بنجاح`,
        data: {
          attachments: newAttachments,
          totalAttachments: updatedAttachments.length
        }
      });

    } catch (error) {
      console.error('Upload error:', error);
      
      // Delete uploaded files on error
      if (req.files) {
        req.files.forEach(file => {
          deleteFile(file.path);
        });
      }

      res.status(500).json({
        success: false,
        message: 'خطأ في حفظ معلومات الملفات'
      });
    }
  });
});

// Get contract attachments
router.get('/contracts/:contractId', authenticateToken, async (req, res) => {
  try {
    const { contractId } = req.params;

    const contract = await Contract.findByPk(contractId);
    if (!contract) {
      return res.status(404).json({
        success: false,
        message: 'العقد غير موجود'
      });
    }

    const attachments = contract.attachments || [];
    
    // Add file existence check
    const attachmentsWithStatus = attachments.map(attachment => {
      const fileInfo = getFileInfo(attachment.filename);
      return {
        ...attachment,
        exists: fileInfo.exists,
        actualSize: fileInfo.size
      };
    });

    res.json({
      success: true,
      data: attachmentsWithStatus
    });

  } catch (error) {
    console.error('Get attachments error:', error);
    res.status(500).json({
      success: false,
      message: 'خطأ في جلب المرفقات'
    });
  }
});

// Download attachment
router.get('/contracts/:contractId/download/:filename', authenticateToken, async (req, res) => {
  try {
    const { contractId, filename } = req.params;

    // Verify contract exists and user has access
    const contract = await Contract.findByPk(contractId);
    if (!contract) {
      return res.status(404).json({
        success: false,
        message: 'العقد غير موجود'
      });
    }

    // Check if file exists in contract attachments
    const attachments = contract.attachments || [];
    const attachment = attachments.find(att => att.filename === filename);
    
    if (!attachment) {
      return res.status(404).json({
        success: false,
        message: 'المرفق غير موجود'
      });
    }

    const filePath = path.join(contractsDir, filename);
    
    // Check if file exists on disk
    if (!fs.existsSync(filePath)) {
      return res.status(404).json({
        success: false,
        message: 'الملف غير موجود على الخادم'
      });
    }

    // Set appropriate headers
    res.setHeader('Content-Disposition', `attachment; filename="${attachment.originalName}"`);
    res.setHeader('Content-Type', attachment.mimetype);

    // Stream the file
    const fileStream = fs.createReadStream(filePath);
    fileStream.pipe(res);

  } catch (error) {
    console.error('Download error:', error);
    res.status(500).json({
      success: false,
      message: 'خطأ في تحميل الملف'
    });
  }
});

// Delete attachment
router.delete('/contracts/:contractId/attachments/:attachmentId', authenticateToken, async (req, res) => {
  try {
    const { contractId, attachmentId } = req.params;

    const contract = await Contract.findByPk(contractId);
    if (!contract) {
      return res.status(404).json({
        success: false,
        message: 'العقد غير موجود'
      });
    }

    const attachments = contract.attachments || [];
    const attachmentIndex = attachments.findIndex(att => att.id.toString() === attachmentId);
    
    if (attachmentIndex === -1) {
      return res.status(404).json({
        success: false,
        message: 'المرفق غير موجود'
      });
    }

    const attachment = attachments[attachmentIndex];
    
    // Delete file from disk
    const filePath = path.join(contractsDir, attachment.filename);
    deleteFile(filePath);

    // Remove from attachments array
    const updatedAttachments = attachments.filter(att => att.id.toString() !== attachmentId);

    // Update contract
    await contract.update({
      attachments: updatedAttachments,
      updatedBy: req.user.id
    });

    res.json({
      success: true,
      message: 'تم حذف المرفق بنجاح',
      data: {
        deletedAttachment: attachment,
        remainingCount: updatedAttachments.length
      }
    });

  } catch (error) {
    console.error('Delete attachment error:', error);
    res.status(500).json({
      success: false,
      message: 'خطأ في حذف المرفق'
    });
  }
});

// Update attachment description
router.put('/contracts/:contractId/attachments/:attachmentId', authenticateToken, async (req, res) => {
  try {
    const { contractId, attachmentId } = req.params;
    const { description } = req.body;

    const contract = await Contract.findByPk(contractId);
    if (!contract) {
      return res.status(404).json({
        success: false,
        message: 'العقد غير موجود'
      });
    }

    const attachments = contract.attachments || [];
    const attachmentIndex = attachments.findIndex(att => att.id.toString() === attachmentId);
    
    if (attachmentIndex === -1) {
      return res.status(404).json({
        success: false,
        message: 'المرفق غير موجود'
      });
    }

    // Update attachment description
    attachments[attachmentIndex].description = description || '';
    attachments[attachmentIndex].updatedAt = new Date().toISOString();

    // Update contract
    await contract.update({
      attachments: attachments,
      updatedBy: req.user.id
    });

    res.json({
      success: true,
      message: 'تم تحديث وصف المرفق بنجاح',
      data: attachments[attachmentIndex]
    });

  } catch (error) {
    console.error('Update attachment error:', error);
    res.status(500).json({
      success: false,
      message: 'خطأ في تحديث المرفق'
    });
  }
});

module.exports = router;
