// Database Initialization Script

const { sequelize } = require('../config/database');
const { User, Contract, Scene, Settings } = require('../models');
const bcrypt = require('bcryptjs');

async function initializeDatabase() {
    try {
        console.log('🔄 بدء تهيئة قاعدة البيانات...');

        // Test database connection
        await sequelize.authenticate();
        console.log('✅ تم الاتصال بقاعدة البيانات بنجاح');

        // Sync all models (create tables)
        await sequelize.sync({ force: false, alter: true });
        console.log('✅ تم إنشاء/تحديث جداول قاعدة البيانات');

        // Create default admin user
        await createDefaultAdmin();

        // Initialize default settings
        await initializeDefaultSettings();

        // Create sample data (optional)
        if (process.argv.includes('--sample-data')) {
            await createSampleData();
        }

        console.log('🎉 تم تهيئة قاعدة البيانات بنجاح!');
        console.log('\n📋 معلومات المدير الافتراضي:');
        console.log('   اسم المستخدم: admin');
        console.log('   كلمة المرور: admin123');
        console.log('   ⚠️  يرجى تغيير كلمة المرور بعد تسجيل الدخول\n');

    } catch (error) {
        console.error('❌ خطأ في تهيئة قاعدة البيانات:', error);
        process.exit(1);
    }
}

async function createDefaultAdmin() {
    try {
        // Check if admin user already exists
        const existingAdmin = await User.findOne({
            where: { userName: 'admin' }
        });

        if (existingAdmin) {
            console.log('ℹ️  المدير الافتراضي موجود بالفعل');
            return;
        }

        // Create default admin user
        const adminUser = await User.create({
            userName: 'admin',
            fullName: 'مدير النظام',
            phone: '1234567890',
            password: 'admin123',
            userType: 'admin',
            permissions: {
                contracts: {
                    add: true,
                    edit: true,
                    view: true,
                    delete: true,
                    print: true
                },
                scenes: {
                    add: true,
                    edit: true,
                    view: true,
                    delete: true,
                    print: true
                }
            },
            isActive: true
        });

        console.log('✅ تم إنشاء المدير الافتراضي');
    } catch (error) {
        console.error('❌ خطأ في إنشاء المدير الافتراضي:', error);
        throw error;
    }
}

async function initializeDefaultSettings() {
    try {
        const defaultSettings = [
            // Contract settings
            {
                key: 'next_contract_number',
                value: '1',
                type: 'number',
                description: 'رقم العقد التالي',
                category: 'contracts'
            },
            {
                key: 'contract_number_prefix',
                value: 'C-',
                type: 'string',
                description: 'بادئة رقم العقد',
                category: 'contracts'
            },
            
            // Print settings
            {
                key: 'print_header',
                value: 'نظام إدارة عقود الصيانة',
                type: 'string',
                description: 'رأس الصفحة للطباعة',
                category: 'print'
            },
            {
                key: 'print_footer',
                value: 'جميع الحقوق محفوظة',
                type: 'string',
                description: 'تذييل الصفحة للطباعة',
                category: 'print'
            },
            {
                key: 'print_margins',
                value: JSON.stringify({
                    top: '20mm',
                    bottom: '20mm',
                    left: '15mm',
                    right: '15mm'
                }),
                type: 'json',
                description: 'هوامش الصفحة للطباعة',
                category: 'print'
            },
            
            // Backup settings
            {
                key: 'backup_enabled',
                value: 'true',
                type: 'boolean',
                description: 'تفعيل النسخ الاحتياطي التلقائي',
                category: 'backup'
            },
            {
                key: 'backup_schedule',
                value: '0 2 * * *',
                type: 'string',
                description: 'جدولة النسخ الاحتياطي (cron format)',
                category: 'backup'
            },
            {
                key: 'backup_retention_days',
                value: '30',
                type: 'number',
                description: 'عدد أيام الاحتفاظ بالنسخ الاحتياطية',
                category: 'backup'
            },
            
            // System settings
            {
                key: 'system_name',
                value: 'نظام إدارة عقود الصيانة',
                type: 'string',
                description: 'اسم النظام',
                category: 'system'
            },
            {
                key: 'max_file_size',
                value: '10485760',
                type: 'number',
                description: 'الحد الأقصى لحجم الملف (بايت)',
                category: 'system'
            },
            {
                key: 'allowed_file_types',
                value: JSON.stringify(['jpg', 'jpeg', 'png', 'gif', 'pdf', 'doc', 'docx', 'xls', 'xlsx']),
                type: 'json',
                description: 'أنواع الملفات المسموحة',
                category: 'system'
            }
        ];

        for (const settingData of defaultSettings) {
            await Settings.findOrCreate({
                where: { key: settingData.key },
                defaults: settingData
            });
        }

        console.log('✅ تم تهيئة الإعدادات الافتراضية');
    } catch (error) {
        console.error('❌ خطأ في تهيئة الإعدادات:', error);
        throw error;
    }
}

async function createSampleData() {
    try {
        console.log('🔄 إنشاء بيانات تجريبية...');

        // Get admin user
        const adminUser = await User.findOne({ where: { userName: 'admin' } });

        // Create sample contracts
        const sampleContracts = [
            {
                contractNumber: 'C-000001',
                contractDate: '2024-01-15',
                secondParty: 'شركة الأمان للحراسة',
                location: 'الرياض - حي الملز - شارع الملك فهد',
                ownerName: 'أحمد محمد السعيد',
                ownerNationalID: '1234567890',
                ownerPhoneNumber: '0501234567',
                camerasType: 'IP كاميرات عالية الدقة',
                camerasResolution: '4K Ultra HD',
                innerCamerasNumber: 8,
                outerCamerasNumber: 4,
                storageCapacity: '2TB هارد ديسك',
                startsFrom: '2024-02-01',
                endsAt: '2025-01-31',
                notes: 'عقد صيانة شامل لمدة سنة واحدة',
                status: 'active',
                createdBy: adminUser.id
            },
            {
                contractNumber: 'C-000002',
                contractDate: '2024-02-10',
                secondParty: 'مجمع الأعمال التجاري',
                location: 'جدة - حي الروضة - طريق الملك عبدالعزيز',
                ownerName: 'فاطمة علي الزهراني',
                ownerNationalID: '2345678901',
                ownerPhoneNumber: '0502345678',
                camerasType: 'كاميرات مراقبة ذكية',
                camerasResolution: 'Full HD 1080p',
                innerCamerasNumber: 12,
                outerCamerasNumber: 6,
                storageCapacity: '4TB هارد ديسك',
                startsFrom: '2024-03-01',
                endsAt: '2025-02-28',
                notes: 'عقد صيانة مع خدمة المراقبة عن بعد',
                status: 'active',
                createdBy: adminUser.id
            }
        ];

        const createdContracts = [];
        for (const contractData of sampleContracts) {
            const contract = await Contract.create(contractData);
            createdContracts.push(contract);
        }

        // Create sample scenes
        const sampleScenes = [
            {
                contractId: createdContracts[0].id,
                date: '2024-02-15',
                action: 'صيانة',
                applicantName: 'أحمد محمد السعيد',
                applicantPhone: '0501234567',
                technician: 'محمد أحمد الفني',
                startTime: '09:00',
                endTime: '11:30',
                status: 'completed',
                workDescription: 'فحص شامل لجميع الكاميرات وتنظيف العدسات',
                notes: 'تم العمل بنجاح وجميع الكاميرات تعمل بكفاءة',
                createdBy: adminUser.id
            },
            {
                contractId: createdContracts[1].id,
                date: '2024-03-05',
                action: 'تركيب',
                applicantName: 'فاطمة علي الزهراني',
                applicantPhone: '0502345678',
                technician: 'علي سعد التقني',
                startTime: '08:00',
                endTime: '16:00',
                status: 'completed',
                workDescription: 'تركيب كاميرات جديدة في المداخل الرئيسية',
                notes: 'تم التركيب وفقاً للمواصفات المطلوبة',
                createdBy: adminUser.id
            },
            {
                contractId: createdContracts[0].id,
                date: '2024-03-20',
                action: 'إصلاح',
                applicantName: 'أحمد محمد السعيد',
                applicantPhone: '0501234567',
                status: 'pending',
                workDescription: 'إصلاح كاميرا معطلة في الموقف',
                notes: 'بلاغ عن عطل في إحدى الكاميرات',
                createdBy: adminUser.id
            }
        ];

        for (const sceneData of sampleScenes) {
            await Scene.create(sceneData);
        }

        console.log('✅ تم إنشاء البيانات التجريبية');
    } catch (error) {
        console.error('❌ خطأ في إنشاء البيانات التجريبية:', error);
        throw error;
    }
}

// Run initialization if this script is executed directly
if (require.main === module) {
    initializeDatabase()
        .then(() => {
            process.exit(0);
        })
        .catch((error) => {
            console.error('❌ فشل في تهيئة قاعدة البيانات:', error);
            process.exit(1);
        });
}

module.exports = {
    initializeDatabase,
    createDefaultAdmin,
    initializeDefaultSettings,
    createSampleData
};
