// Dashboard Page Handler

class DashboardManager {
    constructor() {
        this.charts = {};
        this.refreshInterval = null;
    }

    async render() {
        const content = `
            <div class="row mb-4">
                <div class="col-12">
                    <h2 class="mb-0">
                        <i class="fas fa-tachometer-alt me-2"></i>
                        لوحة التحكم
                    </h2>
                    <p class="text-muted">نظرة عامة على النظام والإحصائيات</p>
                </div>
            </div>

            <!-- Statistics Cards -->
            <div class="row mb-4" id="statsCards">
                <!-- Stats will be loaded here -->
            </div>

            <!-- Charts Row -->
            <div class="row mb-4">
                <div class="col-lg-8">
                    <div class="card">
                        <div class="card-header">
                            <h5 class="mb-0">
                                <i class="fas fa-chart-line me-2"></i>
                                إحصائيات شهرية
                            </h5>
                        </div>
                        <div class="card-body">
                            <canvas id="monthlyChart" height="300"></canvas>
                        </div>
                    </div>
                </div>
                <div class="col-lg-4">
                    <div class="card">
                        <div class="card-header">
                            <h5 class="mb-0">
                                <i class="fas fa-chart-pie me-2"></i>
                                توزيع حالات العقود
                            </h5>
                        </div>
                        <div class="card-body">
                            <canvas id="contractStatusChart" height="300"></canvas>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Recent Activities -->
            <div class="row">
                <div class="col-lg-6">
                    <div class="card">
                        <div class="card-header">
                            <h5 class="mb-0">
                                <i class="fas fa-file-contract me-2"></i>
                                العقود الحديثة
                            </h5>
                        </div>
                        <div class="card-body">
                            <div id="recentContracts">
                                <!-- Recent contracts will be loaded here -->
                            </div>
                        </div>
                    </div>
                </div>
                <div class="col-lg-6">
                    <div class="card">
                        <div class="card-header">
                            <h5 class="mb-0">
                                <i class="fas fa-tools me-2"></i>
                                المشاهد الحديثة
                            </h5>
                        </div>
                        <div class="card-body">
                            <div id="recentScenes">
                                <!-- Recent scenes will be loaded here -->
                            </div>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Quick Actions -->
            <div class="row mt-4">
                <div class="col-12">
                    <div class="card">
                        <div class="card-header">
                            <h5 class="mb-0">
                                <i class="fas fa-bolt me-2"></i>
                                إجراءات سريعة
                            </h5>
                        </div>
                        <div class="card-body">
                            <div class="row">
                                <div class="col-md-3 mb-3">
                                    <button class="btn btn-primary w-100" onclick="app.navigateTo('contracts', { action: 'add' })">
                                        <i class="fas fa-plus me-2"></i>
                                        إضافة عقد جديد
                                    </button>
                                </div>
                                <div class="col-md-3 mb-3">
                                    <button class="btn btn-success w-100" onclick="app.navigateTo('scenes', { action: 'add' })">
                                        <i class="fas fa-tools me-2"></i>
                                        إضافة مشهد جديد
                                    </button>
                                </div>
                                <div class="col-md-3 mb-3">
                                    <button class="btn btn-info w-100" onclick="app.navigateTo('reports')">
                                        <i class="fas fa-chart-bar me-2"></i>
                                        عرض التقارير
                                    </button>
                                </div>
                                <div class="col-md-3 mb-3">
                                    <button class="btn btn-warning w-100" onclick="dashboard.refreshData()">
                                        <i class="fas fa-sync-alt me-2"></i>
                                        تحديث البيانات
                                    </button>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        `;

        document.getElementById('pageContent').innerHTML = content;
        await this.loadDashboardData();
        this.startAutoRefresh();
    }

    async loadDashboardData() {
        try {
            utils.showLoading();
            const response = await api.getDashboardStats();
            
            if (response.success) {
                this.renderStatistics(response.data);
                this.renderCharts(response.data);
                this.renderRecentActivities(response.data.recent);
            }
        } catch (error) {
            console.error('Dashboard data loading error:', error);
            utils.showToast('خطأ في تحميل بيانات لوحة التحكم', 'danger');
        } finally {
            utils.hideLoading();
        }
    }

    renderStatistics(data) {
        const statsHtml = `
            <div class="col-lg-3 col-md-6 mb-3">
                <div class="stat-card">
                    <div class="stat-icon">
                        <i class="fas fa-file-contract"></i>
                    </div>
                    <div class="stat-number">${data.contracts.total}</div>
                    <div class="stat-label">إجمالي العقود</div>
                </div>
            </div>
            <div class="col-lg-3 col-md-6 mb-3">
                <div class="stat-card success">
                    <div class="stat-icon">
                        <i class="fas fa-check-circle"></i>
                    </div>
                    <div class="stat-number">${data.contracts.active}</div>
                    <div class="stat-label">العقود النشطة</div>
                </div>
            </div>
            <div class="col-lg-3 col-md-6 mb-3">
                <div class="stat-card info">
                    <div class="stat-icon">
                        <i class="fas fa-tools"></i>
                    </div>
                    <div class="stat-number">${data.scenes.total}</div>
                    <div class="stat-label">إجمالي المشاهد</div>
                </div>
            </div>
            <div class="col-lg-3 col-md-6 mb-3">
                <div class="stat-card warning">
                    <div class="stat-icon">
                        <i class="fas fa-clock"></i>
                    </div>
                    <div class="stat-number">${data.scenes.pending}</div>
                    <div class="stat-label">المشاهد المعلقة</div>
                </div>
            </div>
        `;

        document.getElementById('statsCards').innerHTML = statsHtml;
    }

    renderCharts(data) {
        this.renderMonthlyChart(data);
        this.renderContractStatusChart(data.contracts);
    }

    renderMonthlyChart(data) {
        const ctx = document.getElementById('monthlyChart');
        if (!ctx) return;

        // Destroy existing chart if it exists
        if (this.charts.monthly) {
            this.charts.monthly.destroy();
        }

        // Generate last 6 months data
        const months = [];
        const contractsData = [];
        const scenesData = [];

        for (let i = 5; i >= 0; i--) {
            const date = new Date();
            date.setMonth(date.getMonth() - i);
            months.push(date.toLocaleDateString('ar-SA', { month: 'long' }));
            
            // Mock data - in real implementation, this would come from the API
            contractsData.push(Math.floor(Math.random() * 20) + 5);
            scenesData.push(Math.floor(Math.random() * 50) + 10);
        }

        this.charts.monthly = new Chart(ctx, {
            type: 'line',
            data: {
                labels: months,
                datasets: [
                    {
                        label: 'العقود',
                        data: contractsData,
                        borderColor: '#3498db',
                        backgroundColor: 'rgba(52, 152, 219, 0.1)',
                        tension: 0.4,
                        fill: true
                    },
                    {
                        label: 'المشاهد',
                        data: scenesData,
                        borderColor: '#27ae60',
                        backgroundColor: 'rgba(39, 174, 96, 0.1)',
                        tension: 0.4,
                        fill: true
                    }
                ]
            },
            options: {
                responsive: true,
                maintainAspectRatio: false,
                plugins: {
                    legend: {
                        position: 'top',
                        rtl: true
                    }
                },
                scales: {
                    y: {
                        beginAtZero: true,
                        ticks: {
                            stepSize: 1
                        }
                    }
                }
            }
        });
    }

    renderContractStatusChart(contractsData) {
        const ctx = document.getElementById('contractStatusChart');
        if (!ctx) return;

        // Destroy existing chart if it exists
        if (this.charts.contractStatus) {
            this.charts.contractStatus.destroy();
        }

        this.charts.contractStatus = new Chart(ctx, {
            type: 'doughnut',
            data: {
                labels: ['نشط', 'منتهي', 'ملغي'],
                datasets: [{
                    data: [
                        contractsData.active,
                        contractsData.expired,
                        contractsData.total - contractsData.active - contractsData.expired
                    ],
                    backgroundColor: [
                        '#27ae60',
                        '#e74c3c',
                        '#f39c12'
                    ],
                    borderWidth: 2,
                    borderColor: '#fff'
                }]
            },
            options: {
                responsive: true,
                maintainAspectRatio: false,
                plugins: {
                    legend: {
                        position: 'bottom',
                        rtl: true
                    }
                }
            }
        });
    }

    renderRecentActivities(recentData) {
        this.renderRecentContracts(recentData.contracts);
        this.renderRecentScenes(recentData.scenes);
    }

    renderRecentContracts(contracts) {
        const container = document.getElementById('recentContracts');
        
        if (!contracts || contracts.length === 0) {
            container.innerHTML = '<p class="text-muted text-center">لا توجد عقود حديثة</p>';
            return;
        }

        const contractsHtml = contracts.map(contract => `
            <div class="d-flex align-items-center mb-3 p-2 border-bottom">
                <div class="flex-shrink-0">
                    <div class="bg-primary text-white rounded-circle d-flex align-items-center justify-content-center" style="width: 40px; height: 40px;">
                        <i class="fas fa-file-contract"></i>
                    </div>
                </div>
                <div class="flex-grow-1 ms-3">
                    <h6 class="mb-1">${contract.contractNumber}</h6>
                    <p class="mb-1 text-muted small">${contract.secondParty}</p>
                    <small class="text-muted">${utils.formatDate(contract.createdAt)}</small>
                </div>
                <div class="flex-shrink-0">
                    <button class="btn btn-sm btn-outline-primary" onclick="app.navigateTo('contracts', { action: 'view', id: ${contract.id} })">
                        عرض
                    </button>
                </div>
            </div>
        `).join('');

        container.innerHTML = contractsHtml;
    }

    renderRecentScenes(scenes) {
        const container = document.getElementById('recentScenes');
        
        if (!scenes || scenes.length === 0) {
            container.innerHTML = '<p class="text-muted text-center">لا توجد مشاهد حديثة</p>';
            return;
        }

        const scenesHtml = scenes.map(scene => `
            <div class="d-flex align-items-center mb-3 p-2 border-bottom">
                <div class="flex-shrink-0">
                    <div class="bg-success text-white rounded-circle d-flex align-items-center justify-content-center" style="width: 40px; height: 40px;">
                        <i class="fas fa-tools"></i>
                    </div>
                </div>
                <div class="flex-grow-1 ms-3">
                    <h6 class="mb-1">${scene.action}</h6>
                    <p class="mb-1 text-muted small">${scene.contract.contractNumber} - ${scene.applicantName}</p>
                    <small class="text-muted">${utils.formatDate(scene.date)}</small>
                </div>
                <div class="flex-shrink-0">
                    <button class="btn btn-sm btn-outline-success" onclick="app.navigateTo('scenes', { action: 'view', id: ${scene.id} })">
                        عرض
                    </button>
                </div>
            </div>
        `).join('');

        container.innerHTML = scenesHtml;
    }

    async refreshData() {
        utils.showToast('جاري تحديث البيانات...', 'info');
        await this.loadDashboardData();
        utils.showToast('تم تحديث البيانات بنجاح', 'success');
    }

    startAutoRefresh() {
        // Refresh data every 5 minutes
        this.refreshInterval = setInterval(() => {
            this.loadDashboardData();
        }, 5 * 60 * 1000);
    }

    stopAutoRefresh() {
        if (this.refreshInterval) {
            clearInterval(this.refreshInterval);
            this.refreshInterval = null;
        }
    }

    destroy() {
        // Clean up charts
        Object.values(this.charts).forEach(chart => {
            if (chart) chart.destroy();
        });
        this.charts = {};
        
        // Stop auto refresh
        this.stopAutoRefresh();
    }
}

// Create global dashboard manager instance
const dashboard = new DashboardManager();

// Export for use in other modules
window.dashboard = dashboard;
