const { sequelize } = require('../config/database');
const mysql = require('mysql2/promise');
require('dotenv').config();

async function fixEncoding() {
  try {
    console.log('🔧 Starting character encoding fix...');

    // Create direct MySQL connection for ALTER statements
    const connection = await mysql.createConnection({
      host: process.env.DB_HOST,
      port: process.env.DB_PORT,
      user: process.env.DB_USER,
      password: process.env.DB_PASSWORD,
      database: process.env.DB_NAME,
      charset: 'utf8mb4'
    });

    console.log('📊 Checking current character sets...');
    
    // Check current database character set
    const [dbCharset] = await connection.execute(`
      SELECT DEFAULT_CHARACTER_SET_NAME, DEFAULT_COLLATION_NAME 
      FROM information_schema.SCHEMATA 
      WHERE SCHEMA_NAME = ?
    `, [process.env.DB_NAME]);
    
    console.log('Database charset:', dbCharset[0]);

    // Check current table character set
    const [tableCharset] = await connection.execute(`
      SELECT TABLE_NAME, TABLE_COLLATION 
      FROM information_schema.TABLES 
      WHERE TABLE_SCHEMA = ? AND TABLE_NAME = 'scenes'
    `, [process.env.DB_NAME]);
    
    console.log('Table charset:', tableCharset[0]);

    // Check current column character sets
    const [columnCharsets] = await connection.execute(`
      SELECT COLUMN_NAME, CHARACTER_SET_NAME, COLLATION_NAME 
      FROM information_schema.COLUMNS 
      WHERE TABLE_SCHEMA = ? 
      AND TABLE_NAME = 'scenes' 
      AND COLUMN_NAME IN ('action', 'applicantName', 'technician', 'workDescription', 'notes')
    `, [process.env.DB_NAME]);
    
    console.log('Column charsets:', columnCharsets);

    console.log('🔄 Converting table to UTF8MB4...');
    
    // Fix scenes table character set
    await connection.execute('ALTER TABLE scenes CONVERT TO CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci');
    
    console.log('🔄 Fixing column character sets...');
    
    // Fix specific columns that contain Arabic text
    await connection.execute(`
      ALTER TABLE scenes 
      MODIFY COLUMN action VARCHAR(100) CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci,
      MODIFY COLUMN applicantName VARCHAR(100) CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci,
      MODIFY COLUMN technician VARCHAR(100) CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci,
      MODIFY COLUMN workDescription TEXT CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci,
      MODIFY COLUMN notes TEXT CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci
    `);

    // Check if new facility fields exist and fix them
    const [facilityColumns] = await connection.execute(`
      SELECT COLUMN_NAME 
      FROM information_schema.COLUMNS 
      WHERE TABLE_SCHEMA = ? 
      AND TABLE_NAME = 'scenes' 
      AND COLUMN_NAME IN ('facilityName', 'facilityOwnerName', 'facilityLocation')
    `, [process.env.DB_NAME]);

    if (facilityColumns.length > 0) {
      console.log('🔄 Fixing facility field character sets...');
      await connection.execute(`
        ALTER TABLE scenes 
        MODIFY COLUMN facilityName VARCHAR(200) CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci,
        MODIFY COLUMN facilityOwnerName VARCHAR(100) CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci,
        MODIFY COLUMN facilityLocation TEXT CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci
      `);
    }

    console.log('🧹 Cleaning up corrupted data...');
    
    // Update corrupted action values with proper Arabic text
    const actionMappings = {
      '?????': 'صيانة', // Default to maintenance if corrupted
      '????': 'فحص',
      '??????': 'تركيب'
    };

    for (const [corrupted, correct] of Object.entries(actionMappings)) {
      await connection.execute(
        'UPDATE scenes SET action = ? WHERE action = ?',
        [correct, corrupted]
      );
    }

    // Also fix any other corrupted text fields
    await connection.execute(`
      UPDATE scenes 
      SET action = CASE 
        WHEN action REGEXP '^\\?+$' THEN 'صيانة'
        ELSE action 
      END
      WHERE action REGEXP '^\\?+$'
    `);

    console.log('✅ Verifying changes...');
    
    // Verify the changes
    const [verifyColumns] = await connection.execute(`
      SELECT COLUMN_NAME, CHARACTER_SET_NAME, COLLATION_NAME 
      FROM information_schema.COLUMNS 
      WHERE TABLE_SCHEMA = ? 
      AND TABLE_NAME = 'scenes' 
      AND COLUMN_NAME IN ('action', 'applicantName', 'technician', 'workDescription', 'notes')
    `, [process.env.DB_NAME]);
    
    console.log('Updated column charsets:', verifyColumns);

    // Check for any remaining corrupted data
    const [corruptedData] = await connection.execute(`
      SELECT id, action, applicantName 
      FROM scenes 
      WHERE action REGEXP '^\\?+$' OR applicantName REGEXP '^\\?+$'
      LIMIT 5
    `);

    if (corruptedData.length > 0) {
      console.log('⚠️  Still found corrupted data:', corruptedData);
      console.log('You may need to manually update these records.');
    } else {
      console.log('✅ No corrupted data found!');
    }

    await connection.end();
    console.log('✅ Character encoding fix completed successfully!');
    
  } catch (error) {
    console.error('❌ Error fixing encoding:', error);
    process.exit(1);
  }
}

// Run the fix
fixEncoding();
